﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/datazone/DataZoneRequest.h>
#include <aws/datazone/DataZone_EXPORTS.h>
#include <aws/datazone/model/FormInput.h>
#include <aws/datazone/model/PredictionConfiguration.h>

#include <utility>

namespace Aws {
namespace DataZone {
namespace Model {

/**
 */
class CreateAssetRevisionRequest : public DataZoneRequest {
 public:
  AWS_DATAZONE_API CreateAssetRevisionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateAssetRevision"; }

  AWS_DATAZONE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that is provided to ensure the
   * idempotency of the request.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateAssetRevisionRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The revised description of the asset.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateAssetRevisionRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the domain where the asset is being revised.</p>
   */
  inline const Aws::String& GetDomainIdentifier() const { return m_domainIdentifier; }
  inline bool DomainIdentifierHasBeenSet() const { return m_domainIdentifierHasBeenSet; }
  template <typename DomainIdentifierT = Aws::String>
  void SetDomainIdentifier(DomainIdentifierT&& value) {
    m_domainIdentifierHasBeenSet = true;
    m_domainIdentifier = std::forward<DomainIdentifierT>(value);
  }
  template <typename DomainIdentifierT = Aws::String>
  CreateAssetRevisionRequest& WithDomainIdentifier(DomainIdentifierT&& value) {
    SetDomainIdentifier(std::forward<DomainIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The metadata forms to be attached to the asset as part of asset revision.</p>
   */
  inline const Aws::Vector<FormInput>& GetFormsInput() const { return m_formsInput; }
  inline bool FormsInputHasBeenSet() const { return m_formsInputHasBeenSet; }
  template <typename FormsInputT = Aws::Vector<FormInput>>
  void SetFormsInput(FormsInputT&& value) {
    m_formsInputHasBeenSet = true;
    m_formsInput = std::forward<FormsInputT>(value);
  }
  template <typename FormsInputT = Aws::Vector<FormInput>>
  CreateAssetRevisionRequest& WithFormsInput(FormsInputT&& value) {
    SetFormsInput(std::forward<FormsInputT>(value));
    return *this;
  }
  template <typename FormsInputT = FormInput>
  CreateAssetRevisionRequest& AddFormsInput(FormsInputT&& value) {
    m_formsInputHasBeenSet = true;
    m_formsInput.emplace_back(std::forward<FormsInputT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The glossary terms to be attached to the asset as part of asset revision.</p>
   */
  inline const Aws::Vector<Aws::String>& GetGlossaryTerms() const { return m_glossaryTerms; }
  inline bool GlossaryTermsHasBeenSet() const { return m_glossaryTermsHasBeenSet; }
  template <typename GlossaryTermsT = Aws::Vector<Aws::String>>
  void SetGlossaryTerms(GlossaryTermsT&& value) {
    m_glossaryTermsHasBeenSet = true;
    m_glossaryTerms = std::forward<GlossaryTermsT>(value);
  }
  template <typename GlossaryTermsT = Aws::Vector<Aws::String>>
  CreateAssetRevisionRequest& WithGlossaryTerms(GlossaryTermsT&& value) {
    SetGlossaryTerms(std::forward<GlossaryTermsT>(value));
    return *this;
  }
  template <typename GlossaryTermsT = Aws::String>
  CreateAssetRevisionRequest& AddGlossaryTerms(GlossaryTermsT&& value) {
    m_glossaryTermsHasBeenSet = true;
    m_glossaryTerms.emplace_back(std::forward<GlossaryTermsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the asset.</p>
   */
  inline const Aws::String& GetIdentifier() const { return m_identifier; }
  inline bool IdentifierHasBeenSet() const { return m_identifierHasBeenSet; }
  template <typename IdentifierT = Aws::String>
  void SetIdentifier(IdentifierT&& value) {
    m_identifierHasBeenSet = true;
    m_identifier = std::forward<IdentifierT>(value);
  }
  template <typename IdentifierT = Aws::String>
  CreateAssetRevisionRequest& WithIdentifier(IdentifierT&& value) {
    SetIdentifier(std::forward<IdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Te revised name of the asset.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateAssetRevisionRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration of the automatically generated business-friendly metadata
   * for the asset.</p>
   */
  inline const PredictionConfiguration& GetPredictionConfiguration() const { return m_predictionConfiguration; }
  inline bool PredictionConfigurationHasBeenSet() const { return m_predictionConfigurationHasBeenSet; }
  template <typename PredictionConfigurationT = PredictionConfiguration>
  void SetPredictionConfiguration(PredictionConfigurationT&& value) {
    m_predictionConfigurationHasBeenSet = true;
    m_predictionConfiguration = std::forward<PredictionConfigurationT>(value);
  }
  template <typename PredictionConfigurationT = PredictionConfiguration>
  CreateAssetRevisionRequest& WithPredictionConfiguration(PredictionConfigurationT&& value) {
    SetPredictionConfiguration(std::forward<PredictionConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The revision type of the asset.</p>
   */
  inline const Aws::String& GetTypeRevision() const { return m_typeRevision; }
  inline bool TypeRevisionHasBeenSet() const { return m_typeRevisionHasBeenSet; }
  template <typename TypeRevisionT = Aws::String>
  void SetTypeRevision(TypeRevisionT&& value) {
    m_typeRevisionHasBeenSet = true;
    m_typeRevision = std::forward<TypeRevisionT>(value);
  }
  template <typename TypeRevisionT = Aws::String>
  CreateAssetRevisionRequest& WithTypeRevision(TypeRevisionT&& value) {
    SetTypeRevision(std::forward<TypeRevisionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_domainIdentifier;
  bool m_domainIdentifierHasBeenSet = false;

  Aws::Vector<FormInput> m_formsInput;
  bool m_formsInputHasBeenSet = false;

  Aws::Vector<Aws::String> m_glossaryTerms;
  bool m_glossaryTermsHasBeenSet = false;

  Aws::String m_identifier;
  bool m_identifierHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  PredictionConfiguration m_predictionConfiguration;
  bool m_predictionConfigurationHasBeenSet = false;

  Aws::String m_typeRevision;
  bool m_typeRevisionHasBeenSet = false;
};

}  // namespace Model
}  // namespace DataZone
}  // namespace Aws
