"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.IrcPoolClient = void 0;
const ioredis_1 = __importDefault(require("ioredis"));
const IrcClientRedisState_1 = require("./IrcClientRedisState");
const RedisIrcConnection_1 = require("./RedisIrcConnection");
const types_1 = require("./types");
const matrix_appservice_bridge_1 = require("matrix-appservice-bridge");
const stream_1 = require("stream");
const CommandReader_1 = require("./CommandReader");
const log = new matrix_appservice_bridge_1.Logger('IrcPoolClient');
const CONNECTION_TIMEOUT = 40000;
const MAX_MISSED_HEARTBEATS = 5;
class IrcPoolClient extends stream_1.EventEmitter {
    redis;
    connections = new Map();
    shouldRun = true;
    missedHeartbeats = 0;
    heartbeatInterval;
    commandReader;
    cmdReader;
    constructor(url) {
        super();
        this.redis = new ioredis_1.default(url, {
            lazyConnect: true,
        });
        this.redis.on('connecting', () => {
            log.debug('Connecting to', url);
        });
        this.cmdReader = new ioredis_1.default(url, {
            lazyConnect: true,
        });
        this.commandReader = new CommandReader_1.RedisCommandReader(this.cmdReader, types_1.REDIS_IRC_POOL_COMMAND_OUT_STREAM, this.handleStreamCommand.bind(this));
    }
    async sendCommand(type, payload) {
        await this.redis.xadd(types_1.REDIS_IRC_POOL_COMMAND_IN_STREAM, "*", type, JSON.stringify({
            origin_ts: Date.now(),
            info: payload,
        }));
        log.debug(`Sent command in ${type}: ${payload}`);
    }
    async *getPreviouslyConnectedClients() {
        let count = 0;
        for (const [clientId, clientAddressPair] of Object.entries(await this.redis.hgetall(types_1.REDIS_IRC_POOL_CONNECTIONS))) {
            const [, address, portStr] = /(.+):(\d+)/.exec(clientAddressPair) || [];
            // Doing this here allows us to frontload the work that would be done in createOrGetIrcSocket
            const state = await IrcClientRedisState_1.IrcClientRedisState.create(this.redis, clientId, false);
            const connection = new RedisIrcConnection_1.RedisIrcConnection(this, clientId, state);
            const port = parseInt(portStr);
            connection.setConnectionInfo({ localPort: port, localIp: address, clientId });
            this.connections.set(clientId, Promise.resolve(connection));
            yield connection;
            count++;
        }
        log.info(`Found ${count} previously connected clients`);
    }
    async createOrGetIrcSocket(clientId, netOpts) {
        const existingConnection = this.connections.get(clientId);
        if (existingConnection) {
            log.warn(`Re-requested ${clientId} within a session, which might indicate a logic error`);
            return existingConnection;
        }
        log.info(`Requesting new client ${clientId}`);
        // Critical section: Do not await here, do any async logic in `clientPromise`.
        // Check to see we are already connected.
        let isConnected = false;
        const clientPromise = (async () => {
            isConnected = (await this.redis.hget(types_1.REDIS_IRC_POOL_CONNECTIONS, clientId)) !== null;
            // NOTE: Bandaid solution
            const clientState = await IrcClientRedisState_1.IrcClientRedisState.create(this.redis, clientId, !isConnected);
            return new RedisIrcConnection_1.RedisIrcConnection(this, clientId, clientState);
        })();
        this.connections.set(clientId, clientPromise);
        const client = await clientPromise;
        try {
            if (!isConnected) {
                log.info(`Requesting new connection for ${clientId}`);
                await this.sendCommand(types_1.InCommandType.Connect, netOpts);
                // Wait to be told we connected.
            }
            else {
                log.info(`${clientId} is still connected, not requesting connection`);
                await this.sendCommand(types_1.InCommandType.ConnectionPing, { clientId });
            }
            await new Promise((resolve, reject) => {
                const timeout = setTimeout(() => {
                    log.warn(`Connection ${clientId} timed out`);
                    reject(new Error('Connection timed out'));
                }, CONNECTION_TIMEOUT);
                client.once('connect', () => {
                    clearTimeout(timeout);
                    resolve();
                });
                client.once('not-connected', () => {
                    clearTimeout(timeout);
                    reject(new Error('Client was not connected'));
                });
                client.once('error', (err) => {
                    clearTimeout(timeout);
                    reject(err);
                });
            });
            log.info(`Resolved connection for ${clientId}`);
            return client;
        }
        catch (ex) {
            // Clean up after so we can have another attempt
            this.connections.delete(clientId);
            log.warn(`Failed to create client ${clientId}`, ex);
            throw Error(`Failed to create client ${clientId}: ${ex.message}`);
        }
    }
    async handleStreamCommand(cmdType, payload) {
        const commandType = cmdType;
        let commandData;
        if (typeof payload === 'string' && payload[0] === '{') {
            commandData = JSON.parse(payload);
        }
        else {
            commandData = Buffer.from(payload).subarray(types_1.READ_BUFFER_MAGIC_BYTES.length);
        }
        return this.handleCommand(commandType, commandData);
    }
    async handleCommand(commandTypeOrClientId, commandData) {
        // I apologise about this insanity.
        const clientId = Buffer.isBuffer(commandData) ? commandTypeOrClientId : commandData.info.clientId;
        const connection = await this.connections.get(clientId);
        if (Buffer.isBuffer(commandData)) {
            log.debug(`Got incoming write ${commandTypeOrClientId}  (${commandData.byteLength} bytes)`);
        }
        else {
            log.debug(`Got incoming ${commandTypeOrClientId} for ${commandData.info.clientId}`);
        }
        if (commandTypeOrClientId === types_1.OutCommandType.PoolClosing) {
            log.warn("Pool has closed, killing the bridge");
            this.emit('lostConnection');
            return;
        }
        if (!connection) {
            log.warn(`Got command ${commandTypeOrClientId} but no client was connected`);
            return;
        }
        switch (commandTypeOrClientId) {
            case types_1.OutCommandType.Connected:
                connection.emit('connect');
                connection.setConnectionInfo(commandData.info);
                break;
            case types_1.OutCommandType.Disconnected:
                this.connections.delete(connection.clientId);
                connection.emit('end');
                break;
            case types_1.OutCommandType.NotConnected:
                connection.emit('not-connected');
                break;
            case types_1.OutCommandType.Error:
                connection.emit('error', new Error(commandData.info.error));
                break;
            default:
                // eslint-disable-next-line no-case-declarations
                const buffer = commandData;
                connection.emit('data', buffer);
                break;
        }
    }
    async close() {
        this.commandReader.stop();
        if (!this.shouldRun) {
            // Already killed, just exit.
            log.warn("close called, but pool client is not running");
            return;
        }
        clearInterval(this.heartbeatInterval);
        // Catch these, because it's quite explosive.
        this.redis.quit().catch((ex) => {
            log.warn('Failed to quit redis writer', ex);
        });
        this.cmdReader.quit().catch((ex) => {
            log.warn('Failed to quit redis command reader', ex);
        });
        this.shouldRun = false;
    }
    async checkHeartbeat() {
        const lastHeartbeat = parseInt(await this.redis.get(types_1.REDIS_IRC_POOL_HEARTBEAT_KEY) ?? '0');
        if (lastHeartbeat + types_1.HEARTBEAT_EVERY_MS + 1000 > Date.now()) {
            this.missedHeartbeats = 0;
            return;
        }
        // Server may be down!
        this.missedHeartbeats++;
        log.warn(`Missed heartbeat from pool (current: ${this.missedHeartbeats}, max: ${MAX_MISSED_HEARTBEATS})`);
        if (this.missedHeartbeats >= MAX_MISSED_HEARTBEATS) {
            // Catastrophic failure, we need to kill the bridge.
            this.emit('lostConnection');
        }
    }
    async listen() {
        log.info(`Listening for new commands`);
        await this.cmdReader.connect();
        await this.redis.connect();
        // First, check if the pool is up.
        const lastHeartbeat = parseInt(await this.redis.get(types_1.REDIS_IRC_POOL_HEARTBEAT_KEY) ?? '0');
        if (lastHeartbeat + types_1.HEARTBEAT_EVERY_MS + 1000 < Date.now()) {
            // Heartbeat is stale or missing, might not be running!
            throw Error('IRC pool is not running!');
        }
        const version = parseInt(await this.redis.get(types_1.REDIS_IRC_POOL_VERSION_KEY) ?? '-1');
        if (version < types_1.PROTOCOL_VERSION) {
            // Heartbeat is stale or missing, might not be running!
            throw Error(`IRC pool is running an older version (${version})` +
                `of the protocol than the bridge (${types_1.PROTOCOL_VERSION}). Restart the pool.`);
        }
        this.heartbeatInterval = setInterval(this.checkHeartbeat.bind(this), types_1.HEARTBEAT_EVERY_MS);
        await this.commandReader.start();
    }
}
exports.IrcPoolClient = IrcPoolClient;
//# sourceMappingURL=IrcPoolClient.js.map