/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%              M   M   OOO   GGGGG  RRRR   IIIII  FFFFF  Y   Y                %
%              MM MM  O   O  G      R   R    I    F       Y Y                 %
%              M M M  O   O  G GGG  RRRR     I    FFF      Y                  %
%              M   M  O   O  G   G  R R      I    F        Y                  %
%              M   M   OOO   GGGG   R  R   IIIII  F        Y                  %
%                                                                             %
%                                                                             %
%                        ImageMagick Module Methods                           %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                                March 2000                                   %
%                                                                             %
%                                                                             %
%  Copyright 1999-2004 ImageMagick Studio LLC, a non-profit organization      %
%  dedicated to making software imaging solutions freely available.           %
%                                                                             %
%  You may not use this file except in compliance with the License.  You may  %
%  obtain a copy of the License at                                            %
%                                                                             %
%    http://www.imagemagick.org/www/Copyright.html                            %
%                                                                             %
%  Unless required by applicable law or agreed to in writing, software        %
%  distributed under the License is distributed on an "AS IS" BASIS,          %
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   %
%  See the License for the specific language governing permissions and        %
%  limitations under the License.                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/annotate.h"
#include "magick/attribute.h"
#include "magick/animate.h"
#include "magick/cache.h"
#include "magick/client.h"
#include "magick/coder.h"
#include "magick/color.h"
#include "magick/colorspace.h"
#include "magick/composite.h"
#include "magick/configure.h"
#include "magick/constitute.h"
#include "magick/decorate.h"
#include "magick/delegate.h"
#include "magick/display.h"
#include "magick/draw.h"
#include "magick/effect.h"
#include "magick/enhance.h"
#include "magick/error.h"
#include "magick/fx.h"
#include "magick/gem.h"
#include "magick/geometry.h"
#include "magick/image.h"
#include "magick/list.h"
#include "magick/locale_.h"
#include "magick/log.h"
#include "magick/magic.h"
#include "magick/magick.h"
#include "magick/memory_.h"
#include "magick/module.h"
#include "magick/mogrify.h"
#include "magick/monitor.h"
#include "magick/montage.h"
#include "magick/option.h"
#include "magick/paint.h"
#include "magick/profile.h"
#include "magick/quantize.h"
#include "magick/random.h"
#include "magick/resize.h"
#include "magick/resource_.h"
#include "magick/segment.h"
#include "magick/signature.h"
#include "magick/shear.h"
#include "magick/string_.h"
#include "magick/transform.h"
#include "magick/token.h"
#include "magick/type.h"
#include "magick/utility.h"
#include "magick/version.h"
#include "magick/xwindow.h"

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
+     M o g r i f y I m a g e                                                 %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  MogrifyImage() applies image processing options to an image as prescribed
%  by command line options.
%
%  The format of the MogrifyImage method is:
%
%      MagickBooleanType MogrifyImage(const ImageInfo *image_info,
%        const int argc,char **argv,Image **image)
%
%  A description of each parameter follows:
%
%    o image_info: The image info..
%
%    o argc: Specifies a pointer to an integer describing the number of
%      elements in the argument vector.
%
%    o argv: Specifies a pointer to a text array containing the command line
%      arguments.
%
%    o image: The image.
%
%
*/
MagickExport MagickBooleanType MogrifyImage(const ImageInfo *image_info,
  const int argc,char **argv,Image **image)
{
  ChannelType
    channel;

  char
    *option;

  DrawInfo
    *draw_info;

  GeometryInfo
    geometry_info;

  Image
    *map_image,
    *region_image;

  ImageInfo
    *mogrify_info;

  MagickStatusType
    flags;

  QuantizeInfo
    quantize_info;

  RectangleInfo
    geometry,
    region_geometry;

  register long
    i;

  /*
    Initialize method variables.
  */
  assert(image_info != (ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  assert(image != (Image **) NULL);
  assert((*image)->signature == MagickSignature);
  if ((*image)->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),(*image)->filename);
  if (argc < 0)
    return(MagickTrue);
  for (i=0; i < (long) argc; i++)
    if (strlen(argv[i]) > (MaxTextExtent/2-1))
      ThrowMagickFatalException(OptionFatalError,"OptionLengthExceedsLimit",
        argv[i]);
  mogrify_info=CloneImageInfo(image_info);
  draw_info=CloneDrawInfo(mogrify_info,(DrawInfo *) NULL);
  SetGeometryInfo(&geometry_info);
  GetQuantizeInfo(&quantize_info);
  map_image=(Image *) NULL;
  quantize_info.number_colors=0;
  quantize_info.tree_depth=0;
  quantize_info.dither=MagickTrue;
  if (mogrify_info->monochrome != MagickFalse)
    if (IsMonochromeImage(*image,&(*image)->exception) == MagickFalse)
      {
        quantize_info.number_colors=2;
        quantize_info.tree_depth=8;
        quantize_info.colorspace=GRAYColorspace;
      }
  SetGeometry(*image,&region_geometry);
  region_image=(Image *) NULL;
  channel=(ChannelType) ((long) AllChannels &~ (long) OpacityChannel);
  /*
    Transmogrify the image.
  */
  for (i=0; i < (long) argc; i++)
  {
    option=argv[i];
    if ((strlen(option) <= 1) || ((*option != '-') && (*option != '+')))
      continue;
    switch (*(option+1))
    {
      case 'a':
      {
        if (LocaleCompare("affine",option+1) == 0)
          {
            char
              *p;

            /*
              Draw affine matrix.
            */
            if (*option == '+')
              {
                GetAffineMatrix(&draw_info->affine);
                break;
              }
            p=argv[++i];
            draw_info->affine.sx=strtod(p,&p);
            if (*p ==',')
              p++;
            draw_info->affine.rx=strtod(p,&p);
            if (*p ==',')
              p++;
            draw_info->affine.ry=strtod(p,&p);
            if (*p ==',')
              p++;
            draw_info->affine.sy=strtod(p,&p);
            if (*p ==',')
              p++;
            draw_info->affine.tx=strtod(p,&p);
            if (*p ==',')
              p++;
            draw_info->affine.ty=strtod(p,&p);
            break;
          }
        if (LocaleCompare("annotate",option+1) == 0)
          {
            char
              geometry[MaxTextExtent];

            /*
              Annotate image.
            */
            flags=ParseGeometry(argv[++i],&geometry_info);
            CloneString(&draw_info->text,argv[++i]);
            if ((flags & SigmaValue) == 0)
              geometry_info.sigma=geometry_info.rho;
            if ((flags & XiValue) == 0)
              geometry_info.xi=0.0;
            if ((flags & PsiValue) == 0)
              geometry_info.psi=0.0;
            (void) FormatMagickString(geometry,MaxTextExtent,"%+f%+f",
              geometry_info.xi,geometry_info.psi);
            (void) CloneString(&draw_info->geometry,geometry);
            draw_info->affine.sx=
              cos(DegreesToRadians(fmod(geometry_info.rho,360.0)));
            draw_info->affine.rx=
              sin(DegreesToRadians(fmod(geometry_info.rho,360.0)));
            draw_info->affine.ry=
              (-sin(DegreesToRadians(fmod(geometry_info.sigma,360.0))));
            draw_info->affine.sy=
              cos(DegreesToRadians(fmod(geometry_info.sigma,360.0)));
            AnnotateImage(*image,draw_info);
            break;
          }
        if (LocaleCompare("antialias",option+1) == 0)
          {
            mogrify_info->antialias=(MagickBooleanType) (*option == '-');
            draw_info->stroke_antialias=(MagickBooleanType) (*option == '-');
            draw_info->text_antialias=(MagickBooleanType) (*option == '-');
            break;
          }
        break;
      }
      case 'b':
      {
        if (LocaleCompare("background",option+1) == 0)
          {
            if (*option == '+')
              (void) QueryColorDatabase(BackgroundColor,
                &mogrify_info->background_color,&(*image)->exception);
            else
              (void) QueryColorDatabase(argv[++i],
                &mogrify_info->background_color,&(*image)->exception);
            (*image)->background_color=mogrify_info->background_color;
            break;
          }
        if (LocaleCompare("black-threshold",option+1) == 0)
          {
            /*
              Threshold black image.
            */
            (void) BlackThresholdImage(*image,argv[++i]);
            break;
          }
        if (LocaleCompare("blue-primary",option+1) == 0)
          {
            /*
              Blue chromaticity primary point.
            */
            if (*option == '+')
              {
                (*image)->chromaticity.blue_primary.x=0.0;
                (*image)->chromaticity.blue_primary.y=0.0;
                break;
              }
            flags=ParseGeometry(image_info->density,&geometry_info);
            (*image)->chromaticity.blue_primary.x=geometry_info.rho;
            (*image)->chromaticity.blue_primary.y=geometry_info.sigma;
            if ((flags & SigmaValue) == 0)
              (*image)->chromaticity.blue_primary.y=
                (*image)->chromaticity.blue_primary.x;
            break;
          }
        if (LocaleCompare("blur",option+1) == 0)
          {
            Image
              *blur_image;

            /*
              Gaussian blur image.
            */
            flags=ParseGeometry(argv[++i],&geometry_info);
            if ((flags & SigmaValue) == 0)
              geometry_info.sigma=1.0;
            blur_image=BlurImageChannel(*image,channel,geometry_info.rho,
              geometry_info.sigma,&(*image)->exception);
            if (blur_image == (Image *) NULL)
              break;
            InheritException(&blur_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=blur_image;
            break;
          }
        if (LocaleCompare("border",option+1) == 0)
          {
            Image
              *border_image;

            /*
              Surround image with a border of solid color.
            */
            flags=ParsePageGeometry(*image,argv[++i],&geometry);
            if ((flags & SigmaValue) == 0)
              geometry.height=geometry.width;
            border_image=BorderImage(*image,&geometry,&(*image)->exception);
            if (border_image == (Image *) NULL)
              break;
            InheritException(&border_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=border_image;
            break;
          }
        if (LocaleCompare("bordercolor",option+1) == 0)
          {
            if (*option == '+')
              (void) QueryColorDatabase(BorderColor,&mogrify_info->border_color,
                &(*image)->exception);
            else
              (void) QueryColorDatabase(argv[++i],&mogrify_info->border_color,
                &(*image)->exception);
            (*image)->border_color=mogrify_info->border_color;
            draw_info->border_color=mogrify_info->border_color;
            break;
          }
        if (LocaleCompare("box",option+1) == 0)
          {
            (void) QueryColorDatabase(argv[++i],&draw_info->undercolor,
              &(*image)->exception);
            break;
          }
        break;
      }
      case 'c':
      {
        if (LocaleCompare("channel",option+1) == 0)
          {
            if (*option == '+')
              {
                channel=(ChannelType)
                  ((long) AllChannels &~ (long) OpacityChannel);
                break;
              }
            channel=(ChannelType) ParseChannelOption(argv[++i]);
            break;
          }
        if (LocaleCompare("charcoal",option+1) == 0)
          {
            Image
              *charcoal_image;

            /*
              Charcoal image.
            */
            flags=ParseGeometry(argv[++i],&geometry_info);
            if ((flags & SigmaValue) == 0)
              geometry_info.sigma=1.0;
            charcoal_image=CharcoalImage(*image,geometry_info.rho,
              geometry_info.sigma,&(*image)->exception);
            if (charcoal_image == (Image *) NULL)
              break;
            InheritException(&charcoal_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=charcoal_image;
            break;
          }
        if (LocaleCompare("chop",option+1) == 0)
          {
            Image
              *chop_image;

            /*
              Chop the image.
            */
            (void) ParseGravityGeometry(*image,argv[++i],&geometry);
            chop_image=ChopImage(*image,&geometry,&(*image)->exception);
            if (chop_image == (Image *) NULL)
              break;
            InheritException(&chop_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=chop_image;
            break;
          }
        if (LocaleCompare("clip",option+1) == 0)
          {
            if (*option == '+')
              {
                (void) SetImageClipMask(*image,(Image *) NULL);
                break;
              }
            (void) ClipImage(*image);
            break;
          }
        if (LocaleCompare("clip-path",option+1) == 0)
          {
            (void) ClipPathImage(*image,argv[++i],(MagickBooleanType)
              (*option == '-'));
            break;
          }
        if (LocaleCompare("colorize",option+1) == 0)
          {
            Image
              *colorize_image;

            /*
              Colorize the image.
            */
            colorize_image=ColorizeImage(*image,argv[++i],draw_info->fill,
              &(*image)->exception);
            if (colorize_image == (Image *) NULL)
              break;
            InheritException(&colorize_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=colorize_image;
            break;
          }
        if (LocaleCompare("colors",option+1) == 0)
          {
            quantize_info.number_colors=(unsigned long) atol(argv[++i]);
            break;
          }
        if (LocaleCompare("colorspace",option+1) == 0)
          {
            if (*option == '+')
              {
                mogrify_info->colorspace=RGBColorspace;
                quantize_info.colorspace=mogrify_info->colorspace;
                (void) SetImageColorspace(*image,RGBColorspace);
                break;
              }
            mogrify_info->colorspace=(ColorspaceType)
              ParseMagickOption(MagickColorspaceOptions,MagickFalse,argv[++i]);
            quantize_info.colorspace=mogrify_info->colorspace;
            if (mogrify_info->colorspace == CMYKColorspace)
              {
                (void) SetImageColorspace(*image,CMYKColorspace);
                quantize_info.colorspace=RGBColorspace;
              }
            if (mogrify_info->colorspace == GRAYColorspace)
              {
                (void) SetImageColorspace(*image,GRAYColorspace);
                quantize_info.colorspace=GRAYColorspace;
              }
            if (mogrify_info->colorspace == RGBColorspace)
              {
                (void) SetImageColorspace(*image,RGBColorspace);
                quantize_info.colorspace=RGBColorspace;
              }
            break;
          }
        if (LocaleCompare("comment",option+1) == 0)
          {
            if (*option == '-')
              i++;
            break;
          }
        if (LocaleCompare("compose",option+1) == 0)
          {
            if (*option == '+')
              {
                (*image)->compose=OverCompositeOp;
                break;
              }
            (*image)->compose=(CompositeOperator)
              ParseMagickOption(MagickCompositeOptions,MagickFalse,argv[++i]);
            break;
          }
        if (LocaleCompare("compress",option+1) == 0)
          {
            if (*option == '+')
              {
                (*image)->compression=UndefinedCompression;
                break;
              }
            (*image)->compression=(CompressionType)
              ParseMagickOption(MagickCompressionOptions,MagickFalse,argv[++i]);
            break;
          }
        if (LocaleCompare("contrast",option+1) == 0)
          {
            (void) ContrastImage(*image,(MagickBooleanType) (*option == '-'));
            break;
          }
        if (LocaleCompare("convolve",option+1) == 0)
          {
            char
              *p,
              token[MaxTextExtent];

            double
              *kernel;

            register long
              x;

            unsigned long
              order;

            /*
              Convolve image.
            */
            p=argv[++i];
            for (x=0; *p != '\0'; x++)
            {
              GetToken(p,&p,token);
              if (*token == ',')
                GetToken(p,&p,token);
            }
            order=(unsigned long) sqrt((double) x+1.0);
            kernel=(double *) AcquireMagickMemory(order*order*sizeof(double));
            if (kernel == (double *) NULL)
              ThrowMagickFatalException(ResourceLimitFatalError,
                "MemoryAllocationFailed",(*image)->filename);
            p=argv[i];
            for (x=0; *p != '\0'; x++)
            {
              GetToken(p,&p,token);
              if (*token == ',')
                GetToken(p,&p,token);
              kernel[x]=atof(token);
            }
            for ( ; x < (long) (order*order); x++)
              kernel[x]=0.0;
            (void) ConvolveImage(*image,order,kernel,&(*image)->exception);
            kernel=(double *) RelinquishMagickMemory(kernel);
            break;
          }
        if (LocaleCompare("crop",option+1) == 0)
          {
            TransformImage(image,argv[++i],(char *) NULL);
            break;
          }
        if (LocaleCompare("cycle",option+1) == 0)
          {
            /*
              Cycle an image colormap.
            */
            (void) CycleColormapImage(*image,atoi(argv[++i]));
            break;
          }
        break;
      }
      case 'd':
      {
        if (LocaleCompare("delay",option+1) == 0)
          {
            if (*option == '-')
              i++;
            break;
          }
        if (LocaleCompare("debug",option+1) == 0)
          {
            if (*option == '-')
              i++;
            (*image)->debug=IsEventLogging();
            break;
          }
        if (LocaleCompare("define",option+1) == 0)
          {
            if (*option == '-')
              i++;
            break;
          }
        if (LocaleCompare("density",option+1) == 0)
          {
            GeometryInfo
              geometry_info;

            /*
              Set image density.
            */
            (void) CloneString(&mogrify_info->density,argv[++i]);
            (void) CloneString(&draw_info->density,mogrify_info->density);
            flags=ParseGeometry(mogrify_info->density,&geometry_info);
            (*image)->x_resolution=geometry_info.rho;
            (*image)->y_resolution=geometry_info.sigma;
            if ((flags & SigmaValue) == 0)
              (*image)->y_resolution=(*image)->x_resolution;
            break;
          }
        if (LocaleCompare("depth",option+1) == 0)
          {
            mogrify_info->depth=(unsigned long) atol(argv[++i]);
            (*image)->depth=mogrify_info->depth;
            break;
          }
        if (LocaleCompare("despeckle",option+1) == 0)
          {
            Image
              *despeckle_image;

            /*
              Reduce the speckles within an image.
            */
            despeckle_image=DespeckleImage(*image,&(*image)->exception);
            if (despeckle_image == (Image *) NULL)
              break;
            InheritException(&despeckle_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=despeckle_image;
            break;
          }
        if (LocaleCompare("display",option+1) == 0)
          {
            (void) CloneString(&mogrify_info->server_name,argv[++i]);
            (void) CloneString(&draw_info->server_name,
              mogrify_info->server_name);
            break;
          }
        if (LocaleCompare("dispose",option+1) == 0)
          {
            if (*option == '-')
              i++;
            break;
          }
        if (LocaleCompare("dither",option+1) == 0)
          {
            mogrify_info->dither=(MagickBooleanType) (*option == '-');
            quantize_info.dither=(MagickBooleanType) (*option == '-');
            break;
          }
        if (LocaleCompare("draw",option+1) == 0)
          {
            /*
              Draw image.
            */
            (void) CloneString(&draw_info->primitive,argv[++i]);
            (void) DrawImage(*image,draw_info);
            break;
          }
        break;
      }
      case 'e':
      {
        if (LocaleCompare("edge",option+1) == 0)
          {
            Image
              *edge_image;

            /*
              Enhance edges in the image.
            */
            flags=ParseGeometry(argv[++i],&geometry_info);
            if ((flags & SigmaValue) == 0)
              geometry_info.sigma=1.0;
            edge_image=EdgeImage(*image,geometry_info.rho,&(*image)->exception);
            if (edge_image == (Image *) NULL)
              break;
            InheritException(&edge_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=edge_image;
            break;
          }
        if (LocaleCompare("emboss",option+1) == 0)
          {
            Image
              *emboss_image;

            /*
              Gaussian embossen image.
            */
            flags=ParseGeometry(argv[++i],&geometry_info);
            if ((flags & SigmaValue) == 0)
              geometry_info.sigma=1.0;
            emboss_image=EmbossImage(*image,geometry_info.rho,
              geometry_info.sigma,&(*image)->exception);
            if (emboss_image == (Image *) NULL)
              break;
            InheritException(&emboss_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=emboss_image;
            break;
          }
        if (LocaleCompare("encoding",option+1) == 0)
          {
            (void) CloneString(&draw_info->encoding,argv[++i]);
            break;
          }
        if (LocaleCompare("endian",option+1) == 0)
          {
            if (*option == '+')
              {
                mogrify_info->endian=LSBEndian;
                break;
              }
            mogrify_info->endian=(EndianType)
              ParseMagickOption(MagickEndianOptions,MagickFalse,argv[++i]);
            (*image)->endian=mogrify_info->endian;
            break;
          }
        if (LocaleCompare("enhance",option+1) == 0)
          {
            Image
              *enhance_image;

            /*
              Enhance image.
            */
            enhance_image=EnhanceImage(*image,&(*image)->exception);
            if (enhance_image == (Image *) NULL)
              break;
            InheritException(&enhance_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=enhance_image;
            break;
          }
        if (LocaleCompare("equalize",option+1) == 0)
          {
            /*
              Equalize image.
            */
            (void) EqualizeImage(*image);
            break;
          }
        if (LocaleCompare("evaluate",option+1) == 0)
          {
            double
              constant;

            MagickEvaluateOperator
              op;

            op=(MagickEvaluateOperator)
              ParseMagickOption(MagickEvaluateOptions,MagickFalse,argv[++i]);
            constant=StringToDouble(argv[++i],MaxRGB);
            (void) EvaluateImageChannel(*image,channel,op,constant,
              &(*image)->exception);
            break;
          }
        break;
      }
      case 'f':
      {
        if (LocaleCompare("family",option+1) == 0)
          {
            (void) CloneString(&draw_info->family,argv[++i]);
            break;
          }
        if (LocaleCompare("fill",option+1) == 0)
          {
            (void) QueryColorDatabase(argv[++i],&draw_info->fill,
              &(*image)->exception);
            break;
          }
        if (LocaleCompare("filter",option+1) == 0)
          {
            if (*option == '+')
              {
                (*image)->filter=UndefinedFilter;
                break;
              }
            (*image)->filter=(FilterTypes)
              ParseMagickOption(MagickFilterOptions,MagickFalse,argv[++i]);
            break;
          }
        if (LocaleCompare("flip",option+1) == 0)
          {
            Image
              *flip_image;

            /*
              Flip image scanlines.
            */
            flip_image=FlipImage(*image,&(*image)->exception);
            if (flip_image == (Image *) NULL)
              break;
            InheritException(&flip_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=flip_image;
            break;
          }
        if (LocaleCompare("flop",option+1) == 0)
          {
            Image
              *flop_image;

            /*
              Flop image scanlines.
            */
            flop_image=FlopImage(*image,&(*image)->exception);
            if (flop_image == (Image *) NULL)
              break;
            InheritException(&flop_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=flop_image;
            break;
          }
        if (LocaleCompare("font",option+1) == 0)
          {
            (void) CloneString(&mogrify_info->font,argv[++i]);
            (void) CloneString(&draw_info->font,mogrify_info->font);
            break;
          }
        if (LocaleCompare("frame",option+1) == 0)
          {
            FrameInfo
              frame_info;

            Image
              *frame_image;

            /*
              Surround image with an ornamental border.
            */
            (void) ParsePageGeometry(*image,argv[++i],&geometry);
            frame_info.width=geometry.width;
            frame_info.height=geometry.height;
            frame_info.outer_bevel=geometry.x;
            frame_info.inner_bevel=geometry.y;
            frame_info.x=(long) frame_info.width;
            frame_info.y=(long) frame_info.height;
            frame_info.width=(*image)->columns+2*frame_info.width;
            frame_info.height=(*image)->rows+2*frame_info.height;
            frame_image=FrameImage(*image,&frame_info,&(*image)->exception);
            if (frame_image == (Image *) NULL)
              break;
            InheritException(&frame_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=frame_image;
            break;
          }
        if (LocaleCompare("fuzz",option+1) == 0)
          {
            (*image)->fuzz=StringToDouble(argv[++i],MaxRGB);
            break;
          }
        break;
      }
      case 'g':
      {
        if (LocaleCompare("gamma",option+1) == 0)
          {
            if (*option == '+')
              (*image)->gamma=atof(argv[++i]);
            else
              {
                if (strchr(option,',') != (char *) NULL)
                  (void) GammaImage(*image,argv[++i]);
                else
                  (void) GammaImageChannel(*image,channel,atof(argv[++i]));
              }
            break;
          }
        if (LocaleCompare("gaussian",option+1) == 0)
          {
            Image
              *gaussian_image;

            /*
              Gaussian blur image.
            */
            flags=ParseGeometry(argv[++i],&geometry_info);
            if ((flags & SigmaValue) == 0)
              geometry_info.sigma=1.0;
            gaussian_image=GaussianBlurImageChannel(*image,channel,
              geometry_info.rho,geometry_info.sigma,&(*image)->exception);
            if (gaussian_image == (Image *) NULL)
              break;
            InheritException(&gaussian_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=gaussian_image;
            break;
          }
        if (LocaleCompare("geometry",option+1) == 0)
          {
            Image
              *zoom_image;

            /*
              Resize image.
            */
            (void) CloneString(&(*image)->geometry,argv[++i]);
            (void) ParseSizeGeometry(*image,(*image)->geometry,&geometry);
            if ((geometry.width == (*image)->columns) &&
                (geometry.height == (*image)->rows))
              break;
            zoom_image=ZoomImage(*image,geometry.width,geometry.height,
              &(*image)->exception);
            if (zoom_image == (Image *) NULL)
              break;
            InheritException(&zoom_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=zoom_image;
            break;
          }
        if (LocaleCompare("gravity",option+1) == 0)
          {
            if (*option == '+')
              {
                draw_info->gravity=UndefinedGravity;
                (*image)->gravity=UndefinedGravity;
                break;
              }
            (*image)->gravity=(GravityType)
              ParseMagickOption(MagickGravityOptions,MagickFalse,argv[++i]);
            draw_info->gravity=(*image)->gravity;
            break;
          }
        if (LocaleCompare("green-primary",option+1) == 0)
          {
            /*
              Green chromaticity primary point.
            */
            if (*option == '+')
              {
                (*image)->chromaticity.green_primary.x=0.0;
                (*image)->chromaticity.green_primary.y=0.0;
                break;
              }
            flags=ParseGeometry(image_info->density,&geometry_info);
            (*image)->chromaticity.green_primary.x=geometry_info.rho;
            (*image)->chromaticity.green_primary.y=geometry_info.sigma;
            if ((flags & SigmaValue) == 0)
              (*image)->chromaticity.green_primary.y=
                (*image)->chromaticity.green_primary.x;
            break;
          }
        break;
      }
      case 'i':
      {
        if (LocaleCompare("implode",option+1) == 0)
          {
            Image
              *implode_image;

            /*
              Implode image.
            */
            (void) ParseGeometry(argv[++i],&geometry_info);
            implode_image=ImplodeImage(*image,geometry_info.rho,
              &(*image)->exception);
            if (implode_image == (Image *) NULL)
              break;
            InheritException(&implode_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=implode_image;
            break;
          }
        if (LocaleCompare("interlace",option+1) == 0)
          {
            if (*option == '+')
              {
                mogrify_info->interlace=NoInterlace;
                break;
              }
            mogrify_info->interlace=(InterlaceType)
              ParseMagickOption(MagickInterlaceOptions,MagickFalse,argv[++i]);
            break;
          }
        if (LocaleCompare("intent",option+1) == 0)
          {
            if (*option == '+')
              {
                (*image)->rendering_intent=UndefinedIntent;
                break;
              }
            (*image)->rendering_intent=(RenderingIntent)
              ParseMagickOption(MagickIntentOptions,MagickFalse,argv[++i]);
            break;
          }
        break;
      }
      case 'l':
      {
        if (LocaleCompare("label",option+1) == 0)
          {
            if (*option == '-')
              i++;
            break;
          }
        if (LocaleCompare("lat",option+1) == 0)
          {
            Image
              *threshold_image;

            /*
              Local adaptive threshold image.
            */
            flags=ParseGeometry(argv[++i],&geometry_info);
            if ((flags & PercentValue) != 0)
              geometry_info.xi=MaxRGB*geometry_info.xi/100.0;
            threshold_image=AdaptiveThresholdImage(*image,(unsigned long)
              geometry_info.rho,(unsigned long) geometry_info.sigma,
              (long) geometry_info.xi,&(*image)->exception);
            if (threshold_image == (Image *) NULL)
              break;
            InheritException(&threshold_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=threshold_image;
            break;
          }
        if (LocaleCompare("level",option+1) == 0)
          {
            GeometryInfo
              geometry_info;

            MagickRealType
              black_point,
              gamma,
              white_point;

            MagickStatusType
              flags;

            /*
              Parse levels.
            */
            flags=ParseGeometry(argv[++i],&geometry_info);
            black_point=geometry_info.rho;
            white_point=(MagickRealType) MaxRGB;
            if ((flags & SigmaValue) != 0)
              white_point=geometry_info.sigma;
            gamma=1.0;
            if ((flags & XiValue) != 0)
              gamma=geometry_info.xi;
            if ((white_point <= 10.0) && (gamma > 10.0))
              {
                MagickRealType
                  swap;

                swap=gamma;
                gamma=white_point;
                white_point=swap;
              }
            if ((flags & PercentValue) != 0)
              {
                black_point*=(MagickRealType) MaxRGB/100.0f;
                white_point*=(MagickRealType) MaxRGB/100.0f;
              }
            if ((flags & SigmaValue) == 0)
              white_point=MaxRGB-black_point;
            (void) LevelImageChannel(*image,channel,black_point,white_point,
              gamma);
            break;
          }
        if (LocaleCompare("linewidth",option+1) == 0)
          {
            draw_info->stroke_width=atof(argv[++i]);
            break;
          }
        if (LocaleCompare("loop",option+1) == 0)
          {
            /*
              Set image iterations.
            */
            (*image)->iterations=(unsigned long) atol(argv[++i]);
            break;
          }
        break;
      }
      case 'm':
      {
        if (LocaleCompare("map",option+1) == 0)
          {
            /*
              Transform image colors to match this set of colors.
            */
            if (*option == '+')
              break;
            (void) CopyMagickString(mogrify_info->filename,argv[++i],
              MaxTextExtent);
            map_image=ReadImage(mogrify_info,&(*image)->exception);
            if ((*image)->exception.severity != UndefinedException)
              CatchException(&(*image)->exception);
            break;
          }
        if (LocaleCompare("mask",option+1) == 0)
          {
            Image
              *mask;

            long
              y;

            register long
              x;

            register PixelPacket
              *q;

            if (*option == '+')
              {
                /*
                  Remove a clip mask.
                */
                (void) SetImageClipMask(*image,(Image *) NULL);
                break;
              }
            /*
              Set the image clip mask.
            */
            (void) CopyMagickString(mogrify_info->filename,argv[++i],
              MaxTextExtent);
            mask=ReadImage(mogrify_info,&(*image)->exception);
            if ((*image)->exception.severity != UndefinedException)
              CatchException(&(*image)->exception);
            if (mask == (Image *) NULL)
              break;
            for (y=0; y < (long) mask->rows; y++)
            {
              q=GetImagePixels(mask,0,y,mask->columns,1);
              if (q == (PixelPacket *) NULL)
                break;
              for (x=0; x < (long) mask->columns; x++)
              {
                if (mask->matte == MagickFalse)
                  q->opacity=PixelIntensityToQuantum(q);
                q->red=q->opacity;
                q->green=q->opacity;
                q->blue=q->opacity;
                q++;
              }
              if (SyncImagePixels(mask) == MagickFalse)
                break;
            }
            mask->storage_class=DirectClass;
            mask->matte=MagickTrue;
            (void) SetImageClipMask(*image,mask);
            break;
          }
        if (LocaleCompare("matte",option+1) == 0)
          {
            if (*option == '-')
              if ((*image)->matte == MagickFalse)
                SetImageOpacity(*image,OpaqueOpacity);
            (*image)->matte=(MagickBooleanType) (*option == '-');
            break;
          }
        if (LocaleCompare("mattecolor",option+1) == 0)
          {
            if (*option == '+')
              (void) QueryColorDatabase(MatteColor,&mogrify_info->border_color,
                &(*image)->exception);
            else
              (void) QueryColorDatabase(argv[++i],&mogrify_info->matte_color,
                &(*image)->exception);
            (*image)->matte_color=mogrify_info->matte_color;
            break;
          }
        if (LocaleCompare("median",option+1) == 0)
          {
            Image
              *median_image;

            /*
              Median filter image.
            */
            (void) ParseGeometry(argv[++i],&geometry_info);
            median_image=MedianFilterImage(*image,geometry_info.rho,
              &(*image)->exception);
            if (median_image == (Image *) NULL)
              break;
            InheritException(&median_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=median_image;
            break;
          }
        if (LocaleCompare("modulate",option+1) == 0)
          {
            (void) ModulateImage(*image,argv[++i]);
            break;
          }
        if (LocaleCompare("monochrome",option+1) == 0)
          {
            mogrify_info->monochrome=MagickTrue;
            quantize_info.number_colors=2;
            quantize_info.tree_depth=8;
            quantize_info.colorspace=GRAYColorspace;
            break;
          }
        break;
      }
      case 'n':
      {
        if (LocaleCompare("negate",option+1) == 0)
          {
            (void) NegateImageChannel(*image,channel,*option == '+');
            break;
          }
        if (LocaleCompare("noise",option+1) == 0)
          {
            Image
              *noisy_image;

            if (*option != '-')
              noisy_image=AddNoiseImage(*image,(NoiseType) ParseMagickOption(
                MagickNoiseOptions,MagickFalse,argv[++i]),&(*image)->exception);
            else
              {
                (void) ParseGeometry(argv[++i],&geometry_info);
                noisy_image=ReduceNoiseImage(*image,geometry_info.rho,
                  &(*image)->exception);
              }
            if (noisy_image == (Image *) NULL)
              break;
            InheritException(&noisy_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=noisy_image;
            break;
          }
        if (LocaleCompare("normalize",option+1) == 0)
          {
            (void) NormalizeImage(*image);
            break;
          }
        break;
      }
      case 'o':
      {
        if (LocaleCompare("opaque",option+1) == 0)
          {
            PixelPacket
              target;

            target=AcquireOnePixel(*image,0,0,&(*image)->exception);
            (void) QueryColorDatabase(argv[++i],&target,&(*image)->exception);
            (void) OpaqueImage(*image,target,draw_info->fill);
            break;
          }
        if (LocaleCompare("ordered-dither",option+1) == 0)
          {
            /*
              Ordered-dither image.
            */
            (void) RandomThresholdImageChannel(*image,channel,argv[++i],
              &(*image)->exception);
            break;
          }
        if (LocaleCompare("orient",option+1) == 0)
          {
            if (*option == '+')
              {
                (*image)->orientation=UndefinedOrientation;
                mogrify_info->orientation=UndefinedOrientation;
                break;
              }
            mogrify_info->orientation=(OrientationType)
              ParseMagickOption(MagickOrientationOptions,MagickFalse,argv[++i]);
            (*image)->orientation=mogrify_info->orientation;
            break;
          }
        break;
      }
      case 'p':
      {
        if (LocaleCompare("page",option+1) == 0)
          {
            if (*option == '-')
              i++;
            break;
          }
        if (LocaleCompare("paint",option+1) == 0)
          {
            Image
              *paint_image;

            /*
              Oil paint image.
            */
            (void) ParseGeometry(argv[++i],&geometry_info);
            paint_image=OilPaintImage(*image,geometry_info.rho,
              &(*image)->exception);
            if (paint_image == (Image *) NULL)
              break;
            InheritException(&paint_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=paint_image;
            break;
          }
        if (LocaleCompare("pen",option+1) == 0)
          {
            (void) QueryColorDatabase(argv[++i],&draw_info->fill,
              &(*image)->exception);
            break;
          }
        if (LocaleCompare("pointsize",option+1) == 0)
          {
            (void) ParseGeometry(argv[++i],&geometry_info);
            mogrify_info->pointsize=geometry_info.rho;
            draw_info->pointsize=geometry_info.rho;
            break;
          }
        if (LocaleCompare("posterize",option+1) == 0)
          {
            /*
              Posterize image.
            */
            (void) PosterizeImage(*image,(unsigned long) atol(argv[++i]),
              quantize_info.dither);
            break;
          }
        if (LocaleCompare("raise",option+1) == 0)
          {
            /*
              Surround image with a raise of solid color.
            */
            (void) ParsePageGeometry(*image,argv[++i],&geometry);
            (void) RaiseImage(*image,&geometry,(MagickBooleanType)
              (*option == '-'));
            break;
          }
        if (LocaleCompare("random-threshold",option+1) == 0)
          {
            /*
              Threshold image.
            */
            (void) RandomThresholdImageChannel(*image,channel,argv[++i],
              &(*image)->exception);
            break;
          }
        if (LocaleCompare("preview",option+1) == 0)
          {
            Image
              *preview_image;

            PreviewType
              preview;

            /*
              Preview image.
            */
            preview=(PreviewType)
              ParseMagickOption(MagickPreviewOptions,MagickFalse,argv[++i]);
            preview_image=PreviewImage(*image,preview,&(*image)->exception);
            if (preview_image == (Image *) NULL)
              break;
            InheritException(&preview_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=preview_image;
            break;
          }
        if (LocaleCompare("profile",option+1) == 0)
          {
            const char
              *name;

            const StringInfo
              *profile;

            Image
              *profile_image;

            void
              *client_data;

            if (*option == '+')
              {
                /*
                  Remove a profile from the image.
                */
                (void) ProfileImage(*image,argv[++i],
                  (const unsigned char *) NULL,0,MagickTrue);
                break;
              }
            /*
              Associate a profile with the image.
            */
            client_data=mogrify_info->client_data;
            mogrify_info->client_data=(void *) GetImageProfile(*image,"8bim");
            (void) CopyMagickString(mogrify_info->filename,argv[++i],
              MaxTextExtent);
            profile_image=ReadImage(mogrify_info,&(*image)->exception);
            if ((*image)->exception.severity != UndefinedException)
              CatchException(&(*image)->exception);
            if (profile_image == (Image *) NULL)
              break;
            ResetImageProfileIterator(profile_image);
            name=GetNextImageProfile(profile_image);
            while (name != (const char *) NULL)
            {
              profile=GetImageProfile(profile_image,name);
              if (profile != (StringInfo *) NULL)
                (void) ProfileImage(*image,name,profile->datum,(unsigned long)
                  profile->length,MagickFalse);
              name=GetNextImageProfile(profile_image);
            }
            DestroyImage(profile_image);
            mogrify_info->client_data=client_data;
            break;
          }
        break;
      }
      case 'q':
      {
        if (LocaleCompare("quality",option+1) == 0)
          {
            /*
              Set image compression quality.
            */
            mogrify_info->quality=(unsigned long) atol(argv[++i]);
            break;
          }
        break;
      }
      case 'r':
      {
        if (LocaleCompare("radial-blur",option+1) == 0)
          {
            Image
              *blur_image;

            /*
              Radial blur image.
            */
            blur_image=RadialBlurImage(*image,atof(argv[++i]),
              &(*image)->exception);
            if (blur_image == (Image *) NULL)
              break;
            InheritException(&blur_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=blur_image;
            break;
          }
        if (LocaleCompare("raise",option+1) == 0)
          {
            /*
              Surround image with a raise of solid color.
            */
            (void) ParsePageGeometry(*image,argv[++i],&geometry);
            (void) RaiseImage(*image,&geometry,(MagickBooleanType)
              (*option == '-'));
            break;
          }
        if (LocaleCompare("random-threshold",option+1) == 0)
          {
            /*
              Threshold image.
            */
            (void) RandomThresholdImageChannel(*image,channel,argv[++i],
              &(*image)->exception);
            break;
          }
        if (LocaleCompare("red-primary",option+1) == 0)
          {
            /*
              Red chromaticity primary point.
            */
            if (*option == '+')
              {
                (*image)->chromaticity.red_primary.x=0.0;
                (*image)->chromaticity.red_primary.y=0.0;
                break;
              }
            flags=ParseGeometry(image_info->density,&geometry_info);
            (*image)->chromaticity.red_primary.x=geometry_info.rho;
            (*image)->chromaticity.red_primary.y=geometry_info.sigma;
            if ((flags & SigmaValue) == 0)
              (*image)->chromaticity.red_primary.y=
                (*image)->chromaticity.red_primary.x;
            break;
          }
        if (LocaleCompare("region",option+1) == 0)
          {
            Image
              *crop_image;

            if (region_image != (Image *) NULL)
              {
                /*
                  Composite region.
                */
                (void) CompositeImage(region_image,(*image)->matte != MagickFalse ?
                  OverCompositeOp : CopyCompositeOp,*image,region_geometry.x,
                  region_geometry.y);
                InheritException(&region_image->exception,&(*image)->exception);
                DestroyImage(*image);
                *image=region_image;
              }
            if (*option == '+')
              break;
            /*
              Apply transformations to a selected region of the image.
            */
            (void) ParseGravityGeometry(*image,argv[++i],&region_geometry);
            crop_image=CropImage(*image,&region_geometry,&(*image)->exception);
            if (crop_image == (Image *) NULL)
              break;
            region_image=(*image);
            *image=crop_image;
            break;
          }
        if (LocaleCompare("render",option+1) == 0)
          {
            draw_info->render=(MagickBooleanType) (*option == '+');
            break;
          }
        if (LocaleCompare("resample",option+1) == 0)
          {
            Image
              *resample_image;

            MagickRealType
              x_resolution,
              y_resolution;

            unsigned long
              height,
              width;

            /*
              Resize image.
            */
            x_resolution=(*image)->x_resolution;
            y_resolution=(*image)->y_resolution;
            if ((*image)->units == PixelsPerCentimeterResolution)
              {
                x_resolution*=2.54f;
                y_resolution*=2.54f;
              }
            flags=ParseGeometry(argv[++i],&geometry_info);
            if ((flags & SigmaValue) == 0)
              geometry_info.sigma=geometry_info.rho;
            width=(unsigned long) (geometry_info.rho*(*image)->columns/
              (x_resolution == 0.0 ? 72.0 : x_resolution)+0.5);
            height=(unsigned long) (geometry_info.sigma*(*image)->rows/
              (y_resolution == 0.0 ? 72.0 : y_resolution)+0.5);
            resample_image=ResizeImage(*image,width,height,(*image)->filter,
              (*image)->blur,&(*image)->exception);
            if (resample_image == (Image *) NULL)
              break;
            InheritException(&resample_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=resample_image;
            (*image)->x_resolution=geometry_info.rho;
            (*image)->y_resolution=geometry_info.sigma;
            break;
          }
        if (LocaleCompare("resize",option+1) == 0)
          {
            Image
              *resize_image;

            /*
              Resize image.
            */
            (void) ParseSizeGeometry(*image,argv[++i],&geometry);
            if ((geometry.width == (*image)->columns) &&
                (geometry.height == (*image)->rows))
              break;
            resize_image=ResizeImage(*image,geometry.width,geometry.height,
              (*image)->filter,(*image)->blur,&(*image)->exception);
            if (resize_image == (Image *) NULL)
              break;
            InheritException(&resize_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=resize_image;
            break;
          }
        if (LocaleCompare("roll",option+1) == 0)
          {
            Image
              *roll_image;

            /*
              Roll image.
            */
            (void) ParsePageGeometry(*image,argv[++i],&geometry);
            roll_image=RollImage(*image,geometry.x,geometry.y,
              &(*image)->exception);
            if (roll_image == (Image *) NULL)
              break;
            InheritException(&roll_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=roll_image;
            break;
          }
        if (LocaleCompare("rotate",option+1) == 0)
          {
            Image
              *rotate_image;

            /*
              Check for conditional image rotation.
            */
            i++;
            if (strchr(argv[i],'>') != (char *) NULL)
              if ((*image)->columns <= (*image)->rows)
                break;
            if (strchr(argv[i],'<') != (char *) NULL)
              if ((*image)->columns >= (*image)->rows)
                break;
            /*
              Rotate image.
            */
            (void) ParseGeometry(argv[i],&geometry_info);
            rotate_image=RotateImage(*image,geometry_info.rho,
              &(*image)->exception);
            if (rotate_image == (Image *) NULL)
              break;
            InheritException(&rotate_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=rotate_image;
            break;
          }
        break;
      }
      case 's':
      {
        if (LocaleCompare("sample",option+1) == 0)
          {
            Image
              *sample_image;

            /*
              Sample image with pixel replication.
            */
            (void) ParseSizeGeometry(*image,argv[++i],&geometry);
            if ((geometry.width == (*image)->columns) &&
                (geometry.height == (*image)->rows))
              break;
            sample_image=SampleImage(*image,geometry.width,geometry.height,
              &(*image)->exception);
            if (sample_image == (Image *) NULL)
              break;
            InheritException(&sample_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=sample_image;
            break;
          }
        if (LocaleCompare("sampling_factor",option+1) == 0)
          {
            /*
              Set image sampling factor.
            */
            (void) CloneString(&mogrify_info->sampling_factor,argv[++i]);
            break;
          }
        if (LocaleCompare("sans",option+1) == 0)
          if (*option == '-')
            i++;
        if (LocaleCompare("scale",option+1) == 0)
          {
            Image
              *scale_image;

            /*
              Resize image.
            */
            (void) ParseSizeGeometry(*image,argv[++i],&geometry);
            if ((geometry.width == (*image)->columns) &&
                (geometry.height == (*image)->rows))
              break;
            scale_image=ScaleImage(*image,geometry.width,geometry.height,
              &(*image)->exception);
            if (scale_image == (Image *) NULL)
              break;
            InheritException(&scale_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=scale_image;
            break;
          }
        if (LocaleCompare("scene",option+1) == 0)
          {
            (*image)->scene=(unsigned long) atol(argv[++i]);
            break;
          }
        if (LocaleCompare("separate",option+1) == 0)
          {
            (void) SeparateImageChannel(*image,channel);
            break;
          }
        if (LocaleCompare("segment",option+1) == 0)
          {
            /*
              Segment image.
            */
            flags=ParseGeometry(argv[++i],&geometry_info);
            if ((flags & SigmaValue) == 0)
              geometry_info.sigma=1.0;
            (void) SegmentImage(*image,quantize_info.colorspace,
              mogrify_info->verbose,geometry_info.rho,geometry_info.sigma);
            break;
          }
        if (LocaleCompare("shade",option+1) == 0)
          {
            Image
              *shade_image;

            /*
              Shade image.
            */
            flags=ParseGeometry(argv[++i],&geometry_info);
            if ((flags & SigmaValue) == 0)
              geometry_info.sigma=0.0;
            shade_image=ShadeImage(*image,(MagickBooleanType) (*option == '-'),
              geometry_info.rho,geometry_info.sigma,&(*image)->exception);
            if (shade_image == (Image *) NULL)
              break;
            InheritException(&shade_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=shade_image;
            break;
          }
        if (LocaleCompare("sharpen",option+1) == 0)
          {
            Image
              *sharp_image;

            /*
              Sharpen image.
            */
            flags=ParseGeometry(argv[++i],&geometry_info);
            if ((flags & SigmaValue) == 0)
              geometry_info.sigma=1.0;
            sharp_image=SharpenImageChannel(*image,channel,geometry_info.rho,
              geometry_info.sigma,&(*image)->exception);
            if (sharp_image == (Image *) NULL)
              break;
            InheritException(&sharp_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=sharp_image;
            break;
          }
        if (LocaleCompare("shave",option+1) == 0)
          {
            Image
              *shave_image;

            /*
              Shave the image edges.
            */
            (void) ParsePageGeometry(*image,argv[++i],&geometry);
            shave_image=ShaveImage(*image,&geometry,&(*image)->exception);
            if (shave_image == (Image *) NULL)
              break;
            InheritException(&shave_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=shave_image;
            break;
          }
        if (LocaleCompare("shear",option+1) == 0)
          {
            Image
              *shear_image;

            /*
              Shear image.
            */
            flags=ParseGeometry(argv[++i],&geometry_info);
            if ((flags & SigmaValue) == 0)
              geometry_info.sigma=geometry_info.rho;
            shear_image=ShearImage(*image,geometry_info.rho,geometry_info.sigma,
              &(*image)->exception);
            if (shear_image == (Image *) NULL)
              break;
            InheritException(&shear_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=shear_image;
            break;
          }
        if (LocaleCompare("solarize",option+1) == 0)
          {
            double
              threshold;

            threshold=StringToDouble(argv[++i],MaxRGB);
            (void) SolarizeImage(*image,threshold);
            break;
          }
        if (LocaleCompare("splice",option+1) == 0)
          {
            Image
              *splice_image;

            /*
              Splice a solid color into the image.
            */
            (void) ParseGravityGeometry(*image,argv[++i],&geometry);
            switch ((*image)->gravity)
            {
              default:
              case UndefinedGravity:
              case NorthWestGravity:
                break;
              case NorthGravity:
              {
                geometry.x+=geometry.width/2;
                break;
              }
              case NorthEastGravity:
              {
                geometry.x+=geometry.width;
                break;
              }
              case WestGravity:
              {
                geometry.x+=geometry.width/2;
                break;
              }
              case StaticGravity:
              case CenterGravity:
              {
                geometry.x+=geometry.width/2;
                geometry.y+=geometry.height/2;
                break;
              }
              case EastGravity:
              {
                geometry.x+=geometry.width;
                geometry.y+=geometry.height/2;
                break;
              }
              case SouthWestGravity:
              {
                geometry.y+=geometry.height/2;
                break;
              }
              case SouthGravity:
              {
                geometry.x+=geometry.width/2;
                geometry.y+=geometry.height;
                break;
              }
              case SouthEastGravity:
              {
                geometry.x+=geometry.width;
                geometry.y+=geometry.height;
                break;
              }
            }
            splice_image=SpliceImage(*image,&geometry,&(*image)->exception);
            if (splice_image == (Image *) NULL)
              break;
            InheritException(&splice_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=splice_image;
            break;
          }
        if (LocaleCompare("spread",option+1) == 0)
          {
            Image
              *spread_image;

            /*
              Spread an image.
            */
            (void) ParseGeometry(argv[++i],&geometry_info);
            spread_image=SpreadImage(*image,geometry_info.rho,
              &(*image)->exception);
            if (spread_image == (Image *) NULL)
              break;
            InheritException(&spread_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=spread_image;
            break;
          }
        if (LocaleCompare("stretch",option+1) == 0)
          {
            if (*option == '+')
              {
                draw_info->stretch=UndefinedStretch;
                break;
              }
            draw_info->stretch=(StretchType)
              ParseMagickOption(MagickStretchOptions,MagickFalse,argv[++i]);
            break;
          }
        if (LocaleCompare("strip",option+1) == 0)
          {
            /*
              Strip image of profiles and comments.
            */
            (void) StripImage(*image);
            break;
          }
        if (LocaleCompare("stroke",option+1) == 0)
          {
            (void) QueryColorDatabase(argv[++i],&draw_info->stroke,
              &(*image)->exception);
            break;
          }
        if (LocaleCompare("strokewidth",option+1) == 0)
          {
            draw_info->stroke_width=atof(argv[++i]);
            break;
          }
        if (LocaleCompare("style",option+1) == 0)
          {
            if (*option == '+')
              {
                draw_info->style=UndefinedStyle;
                break;
              }
            draw_info->style=(StyleType)
              ParseMagickOption(MagickStyleOptions,MagickFalse,argv[++i]);
            break;
          }
        if (LocaleCompare("support",option+1) == 0)
          {
            if (*option == '-')
              (*image)->blur=atof(argv[++i]);
            else
              (*image)->blur=1.0;
            break;
          }
        if (LocaleCompare("swirl",option+1) == 0)
          {
            Image
              *swirl_image;

            /*
              Swirl image.
            */
            (void) ParseGeometry(argv[++i],&geometry_info);
            swirl_image=SwirlImage(*image,geometry_info.rho,
              &(*image)->exception);
            if (swirl_image == (Image *) NULL)
              break;
            InheritException(&swirl_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=swirl_image;
            break;
          }
        break;
      }
      case 't':
      {
        if (LocaleCompare("threshold",option+1) == 0)
          {
            double
              threshold;

            /*
              Threshold image.
            */
            threshold=StringToDouble(argv[++i],MaxRGB);
            (void) BilevelImageChannel(*image,channel,threshold);
            break;
          }
        if (LocaleCompare("thumbnail",option+1) == 0)
          {
            Image
              *thumbnail_image;

            /*
              Thumbnail image.
            */
            (void) ParseSizeGeometry(*image,argv[++i],&geometry);
            if ((geometry.width == (*image)->columns) &&
                (geometry.height == (*image)->rows))
              break;
            thumbnail_image=ThumbnailImage(*image,geometry.width,
              geometry.height,&(*image)->exception);
            if (thumbnail_image == (Image *) NULL)
              break;
            InheritException(&thumbnail_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=thumbnail_image;
            break;
          }
        if (LocaleCompare("tile",option+1) == 0)
          {
            Image
              *fill_pattern;

            if (*option == '+')
              {
                DestroyImage(draw_info->fill_pattern);
                draw_info->fill_pattern=(Image *) NULL;
                break;
              }
            (void) CopyMagickString(mogrify_info->filename,argv[++i],
              MaxTextExtent);
            fill_pattern=ReadImage(mogrify_info,&(*image)->exception);
            if ((*image)->exception.severity != UndefinedException)
              CatchException(&(*image)->exception);
            if (fill_pattern == (Image *) NULL)
              break;
            draw_info->fill_pattern=
              CloneImage(fill_pattern,0,0,MagickTrue,&(*image)->exception);
            DestroyImage(fill_pattern);
            break;
          }
        if (LocaleCompare("tint",option+1) == 0)
          {
            Image
              *tint_image;

            /*
              Tint the image.
            */
            tint_image=TintImage(*image,argv[++i],draw_info->fill,
              &(*image)->exception);
            if (tint_image == (Image *) NULL)
              break;
            InheritException(&tint_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=tint_image;
            break;
          }
        if (LocaleCompare("transform",option+1) == 0)
          {
            Image
              *transform_image;

            /*
              Affine transform image.
            */
            transform_image=AffineTransformImage(*image,&draw_info->affine,
              &(*image)->exception);
            if (transform_image == (Image *) NULL)
              break;
            InheritException(&transform_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=transform_image;
            break;
          }
        if (LocaleCompare("transparent",option+1) == 0)
          {
            PixelPacket
              target;

            target=AcquireOnePixel(*image,0,0,&(*image)->exception);
            (void) QueryColorDatabase(argv[++i],&target,&(*image)->exception);
            (void) TransparentImage(*image,target,TransparentOpacity);
            break;
          }
        if (LocaleCompare("treedepth",option+1) == 0)
          {
            quantize_info.tree_depth=(unsigned long) atol(argv[++i]);
            break;
          }
        if (LocaleCompare("trim",option+1) == 0)
          {
            Image
              *trim_image;

            trim_image=TrimImage(*image,&(*image)->exception);
            if (trim_image == (Image *) NULL)
              break;
            InheritException(&trim_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=trim_image;
            break;
          }
        if (LocaleCompare("type",option+1) == 0)
          {
            if (*option == '+')
              break;
            (void) SetImageType(*image,(ImageType)
              ParseMagickOption(MagickImageOptions,MagickFalse,argv[++i]));
            break;
          }
        break;
      }
      case 'u':
      {
        if (LocaleCompare("undercolor",option+1) == 0)
          {
            (void) QueryColorDatabase(argv[++i],&draw_info->undercolor,
              &(*image)->exception);
            break;
          }
        if (LocaleCompare("units",option+1) == 0)
          {
            if (*option == '+')
              {
                (*image)->units=UndefinedResolution;
                break;
              }
            (*image)->units=(ResolutionType)
              ParseMagickOption(MagickResolutionOptions,MagickFalse,argv[++i]);
            break;
          }
        if (LocaleCompare("unsharp",option+1) == 0)
          {
            Image
              *unsharp_image;

            /*
              Unsharp mask image.
            */
            flags=ParseGeometry(argv[++i],&geometry_info);
            if ((flags & SigmaValue) == 0)
              geometry_info.sigma=1.0;
            if ((flags & XiValue) == 0)
              geometry_info.xi=1.0;
            if ((flags & PsiValue) == 0)
              geometry_info.psi=(double) ScaleCharToQuantum(5.0);
            if ((flags & PercentValue) != 0)
              geometry_info.psi*=MaxRGB/100.0;
            unsharp_image=UnsharpMaskImage(*image,geometry_info.rho,
              geometry_info.sigma,geometry_info.xi,geometry_info.psi,
              &(*image)->exception);
            if (unsharp_image == (Image *) NULL)
              break;
            InheritException(&unsharp_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=unsharp_image;
            break;
          }
        break;
      }
      case 'v':
      {
        if (LocaleCompare("verbose",option+1) == 0)
          {
            mogrify_info->verbose=(MagickBooleanType) (*option == '-');
            quantize_info.measure_error=(MagickBooleanType) (*option == '-');
            break;
          }
        if (LocaleCompare("version",option+1) == 0)
          break;
        if (LocaleCompare("virtual-pixel",option+1) == 0)
          {
            if (*option == '+')
              {
                (void) SetImageVirtualPixelMethod(*image,
                  UndefinedVirtualPixelMethod);
                break;
              }
            (void) SetImageVirtualPixelMethod(*image,(VirtualPixelMethod)
              ParseMagickOption(MagickVirtualPixelOptions,MagickFalse,argv[++i]));
            break;
          }
        break;
      }
      case 'w':
      {
        if (LocaleCompare("wave",option+1) == 0)
          {
            Image
              *wave_image;

            /*
              Wave image.
            */
            flags=ParseGeometry(argv[++i],&geometry_info);
            if ((flags & SigmaValue) == 0)
              geometry_info.sigma=1.0;
            wave_image=WaveImage(*image,geometry_info.rho,geometry_info.sigma,
              &(*image)->exception);
            if (wave_image == (Image *) NULL)
              break;
            InheritException(&wave_image->exception,&(*image)->exception);
            DestroyImage(*image);
            *image=wave_image;
            break;
          }
        if (LocaleCompare("weight",option+1) == 0)
          {
            draw_info->weight=(unsigned long) atol(argv[++i]);
            if (LocaleCompare(argv[i],"all") == 0)
              draw_info->weight=0;
            if (LocaleCompare(argv[i],"bold") == 0)
              draw_info->weight=700;
            if (LocaleCompare(argv[i],"bolder") == 0)
              if (draw_info->weight <= 800)
                draw_info->weight+=100;
            if (LocaleCompare(argv[i],"lighter") == 0)
              if (draw_info->weight >= 100)
                draw_info->weight-=100;
            if (LocaleCompare(argv[i],"normal") == 0)
              draw_info->weight=400;
            break;
          }
        if (LocaleCompare("white-point",option+1) == 0)
          {
            /*
              White chromaticity point.
            */
            if (*option == '+')
              {
                (*image)->chromaticity.white_point.x=0.0;
                (*image)->chromaticity.white_point.y=0.0;
                break;
              }
            flags=ParseGeometry(image_info->density,&geometry_info);
            (*image)->chromaticity.white_point.x=geometry_info.rho;
            (*image)->chromaticity.white_point.y=geometry_info.sigma;
            if ((flags & SigmaValue) == 0)
              (*image)->chromaticity.white_point.y=
                (*image)->chromaticity.white_point.x;
            break;
          }
        if (LocaleCompare("white-threshold",option+1) == 0)
          {
            /*
              Threshold white image.
            */
            (void) WhiteThresholdImage(*image,argv[++i]);
            break;
          }
        break;
      }
      default:
        break;
    }
  }
  if (region_image != (Image *) NULL)
    {
      /*
        Composite transformed region onto image.
      */
      (void) CompositeImage(region_image,
        (*image)->matte != MagickFalse ? OverCompositeOp : CopyCompositeOp,*image,
        region_geometry.x,region_geometry.y);
      InheritException(&region_image->exception,&(*image)->exception);
      DestroyImage(*image);
      *image=region_image;
    }
  if (quantize_info.number_colors != 0)
    {
      /*
        Reduce the number of colors in the image.
      */
      if (((*image)->storage_class == DirectClass) ||
          (*image)->colors > quantize_info.number_colors)
        (void) QuantizeImage(&quantize_info,*image);
      else
        CompressImageColormap(*image);
    }
  if (map_image != (Image *) NULL)
    {
      (void) MapImage(*image,map_image,quantize_info.dither);
      DestroyImage(map_image);
    }
  /*
    Free resources.
  */
  DestroyDrawInfo(draw_info);
  DestroyImageInfo(mogrify_info);
  return((MagickBooleanType) ((*image)->exception.severity == UndefinedException));
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%    M o g r i f y I m a g e C o m m a n d                                    %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  MogrifyImageCommand() transforms an image or a sequence of images. These
%  transforms include image scaling, image rotation, color reduction, and
%  others. The transmogrified image overwrites the original image.
%
%  The format of the MogrifyImageCommand method is:
%
%      MagickBooleanType MogrifyImageCommand(ImageInfo *image_info,int argc,
%        char **argv,char **metadata,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image_info: The image info.
%
%    o argc: The number of elements in the argument vector.
%
%    o argv: A text array containing the command line arguments.
%
%    o metadata: any metadata is returned here.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/

static void MogrifyUsage(void)
{
  static const char
    *options[]=
    {
      "-affine matrix       affine transform matrix",
      "-annotate geometry text",
      "                     annotate the image with text",
      "-antialias           remove pixel-aliasing",
      "-authenticate value  decrypt image with this password",
      "-background color    background color",
      "-black-threshold value",
      "                     forces all pixels below the threshold into black",
      "-blue-primary point  chromaticity blue primary point",
      "-blur geometry       blur the image",
      "-border geometry     surround image with a border of color",
      "-bordercolor color   border color",
      "-channel type        Red, Green, Blue, Opacity, Index, Cyan, Yellow, ",
      "                     Magenta, Black, or All",
      "-charcoal radius     simulate a charcoal drawing",
      "-chop geometry       remove pixels from the image interior",
      "-colorize value      colorize the image with the fill color",
      "-colors value        preferred number of colors in the image",
      "-colorspace type     alternate image colorspace",
      "-comment string      annotate image with comment",
      "-compress type       image compression type",
      "-contrast            enhance or reduce the image contrast",
      "-convolve coefficients",
      "                     apply a convolution kernel to the image",
      "-crop geometry       preferred size and location of the cropped image",
      "-cycle amount        cycle the image colormap",
      "-debug events        display copious debugging information",
      "-define format:option",
      "                     define one or more image format options",
      "-delay value         display the next image after pausing",
      "-density geometry    horizontal and vertical density of the image",
      "-depth value         image depth",
      "-despeckle           reduce the speckles within an image",
      "-display server      get image or font from this X server",
      "-dispose method      Undefined, None, Background, Previous",
      "-dither              apply Floyd/Steinberg error diffusion to image",
      "-draw string         annotate the image with a graphic primitive",
      "-edge radius         apply a filter to detect edges in the image",
      "-emboss radius       emboss an image",
      "-encoding type       text encoding type",
      "-endian type         LSB or MSB",
      "-enhance             apply a digital filter to enhance a noisy image",
      "-equalize            perform histogram equalization to an image",
      "-evaluate operator value",
      "                     evaluate an arithmetic, relational, or logical expression",
      "-extract geometry    extract area from image",
      "-family name         render text with this font family",
      "-fill color          color to use when filling a graphic primitive",
      "-filter type         use this filter when resizing an image",
      "-flip                flip image in the vertical direction",
      "-flop                flop image in the horizontal direction",
      "-font name           render text with this font",
      "-format type         image format type",
      "-frame geometry      surround image with an ornamental border",
      "-fuzz distance       colors within this distance are considered equal",
      "-gamma value         level of gamma correction",
      "-gaussian geometry   gaussian blur an image",
      "-geometry geometry   perferred size or location of the image",
      "-green-primary point chromaticity green primary point",
      "-implode amount      implode image pixels about the center",
      "-interlace type      None, Line, Plane, or Partition",
      "-help                print program options",
      "-label name          assign a label to an image",
      "-lat geometry        local adaptive thresholding",
      "-level value         adjust the level of image contrast",
      "-limit type value    Area, Disk, Map, or Memory resource limit",
      "-log format          format of debugging information",
      "-loop iterations     add Netscape loop extension to your GIF animation",
      "-map filename        transform image colors to match this set of colors",
      "-mask filename       set the image clip mask",
      "-matte               store matte channel if the image has one",
      "-median radius       apply a median filter to the image",
      "-modulate value      vary the brightness, saturation, and hue",
      "-monochrome          transform image to black and white",
      "-negate              replace every pixel with its complementary color ",
      "-noise radius        add or reduce noise in an image.",
      "-normalize           transform image to span the full range of colors",
      "-opaque color        change this color to the fill color",
      "-ordered-dither NxN",
      "                     ordered dither the image",
      "-orient type         image orientation",
      "-page geometry       size and location of an image canvas",
      "-paint radius        simulate an oil painting",
      "-pointsize value     font point size",
      "-posterize levels    reduce the image to a limited number of color levels",
      "-profile filename    add ICM or IPTC information profile to image",
      "-quality value       JPEG/MIFF/PNG compression level",
      "-raise value         lighten/darken image edges to create a 3-D effect",
      "-random-threshold low,high",
      "                     random threshold the image",
      "-radial-blur angle   radial blur the image",
      "-red-primary point   chromaticity red primary point",
      "-region geometry     apply options to a portion of the image",
      "-resample geometry   change the resolution of an image",
      "-resize geometry     perferred size or location of the image",
      "-roll geometry       roll an image vertically or horizontally",
      "-rotate degrees      apply Paeth rotation to the image",
      "-sample geometry     scale image with pixel sampling",
      "-sampling-factor geometry",
      "                     horizontal and vertical sampling factor",
      "-scale geometry      scale the image",
      "-scene number        image scene number",
      "-seed value          pseudo-random number generator seed value",
      "-segment values      segment an image",
      "-separate            separate an image channel into a grayscale image",
      "-shade degrees       shade the image using a distant light source",
      "-sharpen radius      sharpen the image",
      "-shear geometry      slide one edge of the image along the X or Y axis",
      "-size geometry       width and height of image",
      "-solarize threshold  negate all pixels above the threshold level",
      "-spread amount       displace image pixels by a random amount",
      "-stretch type        render text with this font stretch",
      "-strip               strip image of all profiles and comments",
      "-stroke color        graphic primitive stroke color",
      "-strokewidth value   graphic primitive stroke width",
      "-style type          render text with this font style",
      "-support factor      resize support: > 1.0 is blurry, < 1.0 is sharp",
      "-swirl degrees       swirl image pixels about the center",
      "-texture filename    name of texture to tile onto the image background",
      "-threshold value     threshold the image",
      "-thumbnail geometry  create a thumbnail of the image",
      "-tile filename       tile image when filling a graphic primitive",
      "-tint value          tint the image with the fill color",
      "-transform           affine transform image",
      "-transparent color   make this color transparent within the image",
      "-treedepth value     color tree depth",
      "-trim                trim image edges",
      "-type type           image type",
      "-undercolor color    annotation bounding box color",
      "-units type          PixelsPerInch, PixelsPerCentimeter, or Undefined",
      "-unsharp geometry    sharpen the image",
      "-verbose             print detailed information about the image",
      "-version             print version information",
      "-view                FlashPix viewing transforms",
      "-virtual-pixel method",
      "                     Constant, Edge, Mirror, or Tile",
      "-wave geometry       alter an image along a sine wave",
      "-weight type         render text with this font weight",
      "-white-point point   chromaticity white point",
      "-white-threshold value",
      "                     forces all pixels above the threshold into white",
      (char *) NULL
    };

  const char
    **p;

  (void) printf("Version: %s\n",GetMagickVersion((unsigned long *) NULL));
  (void) printf("Copyright: %s\n\n",GetMagickCopyright());
  (void) printf("Usage: %s [options ...] file [ [options ...] file ...]\n",
    GetClientName());
  (void) printf("\nWhere options include: \n");
  for (p=options; *p != (char *) NULL; p++)
    (void) printf("  %s\n",*p);
  (void) printf(
    "\nBy default, the image format of `file' is determined by its magic\n");
  (void) printf(
    "number.  To specify a particular image format, precede the filename\n");
  (void) printf(
    "with an image format name and a colon (i.e. ps:image) or specify the\n");
  (void) printf(
    "image type as the filename suffix (i.e. image.ps).  Specify 'file' as\n");
  (void) printf("'-' for standard input or output.\n");
  Exit(0);
}

MagickExport MagickBooleanType MogrifyImageCommand(ImageInfo *image_info,
  int argc,char **argv,char **metadata,ExceptionInfo *exception)
{
#define DestroyMogrify() \
{ \
  for (i=0; i < (long) argc; i++) \
    argv[i]=(char *) RelinquishMagickMemory(argv[i]); \
  argv=(char **) RelinquishMagickMemory(argv); \
}
#define ThrowMogrifyException(severity,tag,option) \
{ \
  (void) ThrowMagickException(exception,GetMagickModule(),severity,tag,option);\
  DestroyMogrify(); \
  return(MagickFalse); \
}
#define ThrowMogrifyInvalidArgumentException(option,argument) \
{ \
  (void) ThrowMagickException(exception,GetMagickModule(),OptionError, \
    "InvalidArgument",argument,option); \
  DestroyMogrify(); \
  return(MagickFalse); \
}

  char
    *format,
    *option;

  Image
    *image;

  long
    j;

  register long
    i;

  MagickBooleanType
    global_colormap,
    status;

  /*
    Set defaults.
  */
  assert(image_info != (ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  if (image_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"");
  assert(exception != (ExceptionInfo *) NULL);
  if (argc < 2)
    MogrifyUsage();
  format=(char *) NULL;
  global_colormap=MagickFalse;
  image=NewImageList();
  option=(char *) NULL;
  status=MagickTrue;
  /*
    Parse command line.
  */
  ReadCommandlLine(argc,&argv);
  status=ExpandFilenames(&argc,&argv);
  if (status == MagickFalse)
    ThrowMogrifyException(ResourceLimitError,"MemoryAllocationFailed",
      strerror(errno));
  j=1;
  for (i=1; i < (long) argc; i++)
  {
    option=argv[i];
    if ((strlen(option) == 1) || ((*option != '-') && (*option != '+')))
      {
        /*
          Option is a file name: begin by reading image from specified file.
        */
        (void) CopyMagickString(image_info->filename,argv[i],MaxTextExtent);
        image=ReadImage(image_info,exception);
        if (exception->severity != UndefinedException)
          CatchException(exception);
        status&=(image != (Image *) NULL) &&
          (exception->severity < ErrorException);
        if (image == (Image *) NULL)
          continue;
        if (format != (char *) NULL)
          AppendImageFormat(format,image->filename);
        /*
          Transmogrify image as defined by the image processing options.
        */
        status&=MogrifyImages(image_info,(int) (i-j),argv+j,&image);
        if (global_colormap != MagickFalse)
          (void) MapImages(image,(Image *) NULL,image_info->dither);
        /*
          Write transmogrified image to disk.
        */
        status&=WriteImages(image_info,image,image->filename,&image->exception);
        DestroyImage(image);
        continue;
      }
    switch (*(option+1))
    {
      case 'a':
      {
        if (LocaleCompare("affine",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("annotate",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                i++;
              }
            break;
          }
        if (LocaleCompare("antialias",option+1) == 0)
          {
            image_info->antialias=(MagickBooleanType) (*option == '-');
            break;
          }
        if (LocaleCompare("authenticate",option+1) == 0)
          {
            (void) CloneString(&image_info->authenticate,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                (void) CloneString(&image_info->authenticate,argv[i]);
              }
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'b':
      {
        if (LocaleCompare("background",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                (void) QueryColorDatabase(argv[i],&image_info->background_color,
                  exception);
              }
            break;
          }
        if (LocaleCompare("black-threshold",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("blue-primary",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("blur",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            if (IsGeometry(argv[i]) == MagickFalse)
              ThrowMogrifyInvalidArgumentException(option,argv[i]);
            break;
          }
        if (LocaleCompare("border",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("bordercolor",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                (void) QueryColorDatabase(argv[i],&image_info->border_color,
                  exception);
              }
            break;
          }
        if (LocaleCompare("box",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
              }
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'c':
      {
        if (LocaleCompare("cache",option+1) == 0)
          {
            if (*option == '-')
              {
                unsigned long
                  limit;

                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
                limit=(~0UL);
                if (LocaleCompare("unlimited",argv[i]) != 0)
                  limit=(unsigned long) atol(argv[i]);
                (void) SetMagickResourceLimit(MemoryResource,limit);
                (void) SetMagickResourceLimit(MapResource,2*limit);
              }
            break;
          }
        if (LocaleCompare("channel",option+1) == 0)
          {
            if (*option == '-')
              {
                long
                  channel;

                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                channel=ParseChannelOption(argv[i]);
                if (channel < 0)
                  ThrowMogrifyException(OptionError,"UnrecognizedChannelType",
                    argv[i]);
              }
            break;
          }
        if (LocaleCompare("charcoal",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("chop",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("colorize",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("colors",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
                image_info->colors=(unsigned long) atol(argv[i]);
              }
            break;
          }
        if (LocaleCompare("colorspace",option+1) == 0)
          {
            image_info->colorspace=UndefinedColorspace;
            if (*option == '-')
              {
                long
                  colorspace;

                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                colorspace=ParseMagickOption(MagickColorspaceOptions,MagickFalse,
                  argv[i]);
                if (colorspace < 0)
                  ThrowMogrifyException(OptionError,"UnrecognizedColorspace",
                    argv[i]);
                image_info->colorspace=(ColorspaceType) colorspace;
              }
            break;
          }
        if (LocaleCompare("comment",option+1) == 0)
          {
            if (*option == '+')
              {
                status=DeleteImageOption(image_info,"comment");
                break;
              }
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            status=SetImageOption(image_info,"comment",argv[i]);
            if (status == MagickFalse)
              ThrowMogrifyException(OptionError,"UnrecognizedOption",argv[i]);
            break;
          }
        if (LocaleCompare("compress",option+1) == 0)
          {
            image_info->compression=UndefinedCompression;
            if (*option == '-')
              {
                long
                  compression;

                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                compression=ParseMagickOption(MagickCompressionOptions,MagickFalse,
                  argv[i]);
                if (compression < 0)
                  ThrowMogrifyException(OptionError,
                    "UnrecognizedImageCompression",argv[i]);
              }
            break;
          }
        if (LocaleCompare("contrast",option+1) == 0)
          break;
        if (LocaleCompare("convolve",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("crop",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("cycle",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'd':
      {
        if (LocaleCompare("debug",option+1) == 0)
          {
            (void) SetLogEventMask("None");
            if (*option == '-')
              {
                LogEventType
                  event_mask;

                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                event_mask=SetLogEventMask(argv[i]);
                if (event_mask == UndefinedEvents)
                  ThrowMogrifyException(OptionError,"UnrecognizedEventType",
                    option);
              }
            image_info->debug=IsEventLogging();
            break;
          }
        if (LocaleCompare("define",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            if (*option == '+')
              {
                char
                  *define;

                define=RemoveImageOption(image_info,argv[i]);
                if (define == (char *) NULL)
                  ThrowMogrifyException(OptionError,"NoSuchOption",argv[i]);
                define=(char *) RelinquishMagickMemory(define);
                break;
              }
            status=DefineImageOption(image_info,argv[i]);
            if (status == MagickFalse)
              ThrowMogrifyException(OptionError,"UnrecognizedOption",argv[i]);
            break;
          }
        if (LocaleCompare("delay",option+1) == 0)
          {
            if (*option == '+')
              {
                char
                  *delay;

                delay=RemoveImageOption(image_info,argv[i]);
                if (delay != (char *) NULL)
                  delay=(char *) RelinquishMagickMemory(delay);
                break;
              }
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            if (IsGeometry(argv[i]) == MagickFalse)
              ThrowMogrifyInvalidArgumentException(option,argv[i]);
            status=SetImageOption(image_info,"delay",argv[i]);
            if (status == MagickFalse)
              ThrowMogrifyException(OptionError,"UnrecognizedOption",argv[i]);
            break;
          }
        if (LocaleCompare("density",option+1) == 0)
          {
            (void) CloneString(&image_info->density,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
                (void) CloneString(&image_info->density,argv[i]);
              }
            break;
          }
        if (LocaleCompare("depth",option+1) == 0)
          {
            image_info->depth=QuantumDepth;
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
                image_info->depth=(unsigned long) atol(argv[i]);
              }
            break;
          }
        if (LocaleCompare("despeckle",option+1) == 0)
          break;
        if (LocaleCompare("display",option+1) == 0)
          {
            (void) CloneString(&image_info->server_name,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                (void) CloneString(&image_info->server_name,argv[i]);
              }
            break;
          }
        if (LocaleCompare("dispose",option+1) == 0)
          {
            long
              dispose;

            if (*option == '+')
              {
                char
                  *dispose;

                dispose=RemoveImageOption(image_info,argv[i]);
                if (dispose != (char *) NULL)
                  dispose=(char *) RelinquishMagickMemory(dispose);
                break;
              }
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            dispose=ParseMagickOption(MagickDisposeOptions,MagickFalse,argv[i]);
            if (dispose < 0)
              ThrowMogrifyException(OptionError,"UnrecognizedDisposeMethod",
                argv[i]);
            status=SetImageOption(image_info,"dispose",argv[i]);
            if (status == MagickFalse)
              ThrowMogrifyException(OptionError,"UnrecognizedOption",argv[i]);
            break;
          }
        if (LocaleCompare("dither",option+1) == 0)
          {
            image_info->dither=(MagickBooleanType) (*option == '-');
            break;
          }
        if (LocaleCompare("draw",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
              }
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'e':
      {
        if (LocaleCompare("edge",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("emboss",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("encoding",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
              }
            break;
          }
        if (LocaleCompare("endian",option+1) == 0)
          {
            image_info->endian=UndefinedEndian;
            if (*option == '-')
              {
                long
                  endian;

                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                endian=ParseMagickOption(MagickEndianOptions,MagickFalse,argv[i]);
                if (endian < 0)
                  ThrowMogrifyException(OptionError,"UnrecognizedEndianType",
                    argv[i]);
                image_info->endian=(EndianType) endian;
              }
            break;
          }
        if (LocaleCompare("enhance",option+1) == 0)
          break;
        if (LocaleCompare("equalize",option+1) == 0)
          break;
        if (LocaleCompare("evaluate",option+1) == 0)
          {
            if (*option == '-')
              {
                long
                  op;

                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                op=ParseMagickOption(MagickEvaluateOptions,MagickFalse,argv[i]);
                if (op < 0)
                  ThrowMogrifyException(OptionError,
                    "UnrecognizedEvaluateOperator",argv[i]);
                i++;
                if (i == (long) (argc-1))
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("extract",option+1) == 0)
          {
            (void) CloneString(&image_info->extract,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
                (void) CloneString(&image_info->extract,argv[i]);
              }
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'f':
      {
        if (LocaleCompare("family",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) (argc-1))
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
              }
            break;
          }
        if (LocaleCompare("fill",option+1) == 0)
          {
            (void) QueryColorDatabase("none",&image_info->pen,exception);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                (void) QueryColorDatabase(argv[i],&image_info->pen,exception);
              }
            break;
          }
        if (LocaleCompare("filter",option+1) == 0)
          {
            if (*option == '-')
              {
                long
                  filter;

                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                filter=ParseMagickOption(MagickFilterOptions,MagickFalse,argv[i]);
                if (filter < 0)
                  ThrowMogrifyException(OptionError,"UnrecognizedImageFilter",
                    argv[i]);
              }
            break;
          }
        if (LocaleCompare("flip",option+1) == 0)
          break;
        if (LocaleCompare("flop",option+1) == 0)
          break;
        if (LocaleCompare("font",option+1) == 0)
          {
            (void) CloneString(&image_info->font,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                (void) CloneString(&image_info->font,argv[i]);
              }
            break;
          }
        if (LocaleCompare("format",option+1) == 0)
          {
            (void) CloneString(&format,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                (void) CloneString(&format,argv[i]);
                (void) CopyMagickString(image_info->filename,format,
                  MaxTextExtent);
                (void) ConcatenateMagickString(image_info->filename,":",
                  MaxTextExtent);
                (void) SetImageInfo(image_info,MagickFalse,exception);
                if (*image_info->magick == '\0')
                  ThrowMogrifyException(OptionError,
                    "UnrecognizedImageFormat",format);
              }
            break;
          }
        if (LocaleCompare("frame",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("fuzz",option+1) == 0)
          {
            image_info->fuzz=0.0;
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
                image_info->fuzz=StringToDouble(argv[i],MaxRGB);
              }
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'g':
      {
        if (LocaleCompare("gamma",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            if (IsGeometry(argv[i]) == MagickFalse)
              ThrowMogrifyInvalidArgumentException(option,argv[i]);
            break;
          }
        if (LocaleCompare("gaussian",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            if (IsGeometry(argv[i]) == MagickFalse)
              ThrowMogrifyInvalidArgumentException(option,argv[i]);
            break;
          }
        if (LocaleCompare("geometry",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("gravity",option+1) == 0)
          {
            if (*option == '-')
              {
                long
                  gravity;

                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                gravity=ParseMagickOption(MagickGravityOptions,MagickFalse,argv[i]);
                if (gravity < 0)
                  ThrowMogrifyException(OptionError,"UnrecognizedGravityType",
                    argv[i]);
                break;
              }
            break;
          }
        if (LocaleCompare("green-primary",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'h':
      {
        if (LocaleCompare("help",option+1) == 0)
          MogrifyUsage();
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'i':
      {
        if (LocaleCompare("implode",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("interlace",option+1) == 0)
          {
            image_info->interlace=UndefinedInterlace;
            if (*option == '-')
              {
                long
                  interlace;

                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                interlace=ParseMagickOption(MagickInterlaceOptions,MagickFalse,
                  argv[i]);
                if (interlace < 0)
                  ThrowMogrifyException(OptionError,"UnrecognizedInterlaceType",
                    argv[i]);
                image_info->interlace=(InterlaceType) interlace;
              }
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'l':
      {
        if (LocaleCompare("label",option+1) == 0)
          {
            if (*option == '+')
              {
                status=DeleteImageOption(image_info,"label");
                break;
              }
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            status=SetImageOption(image_info,"label",argv[i]);
            if (status == MagickFalse)
              ThrowMogrifyException(OptionError,"UnrecognizedOption",argv[i]);
            break;
          }
        if (LocaleCompare("lat",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("level",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            if (IsGeometry(argv[i]) == MagickFalse)
              ThrowMogrifyInvalidArgumentException(option,argv[i]);
            break;
          }
        if (LocaleCompare("linewidth",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("limit",option+1) == 0)
          {
            if (*option == '-')
              {
                char
                  *type;

                long
                  resource;

                unsigned long
                  limit;

                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                type=argv[i];
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
                limit=(~0UL);
                if (LocaleCompare("unlimited",argv[i]) != 0)
                  limit=(unsigned long) atol(argv[i]);
                resource=ParseMagickOption(MagickResourceOptions,MagickFalse,type);
                if (resource < 0)
                  ThrowMogrifyException(OptionError,
                    "UnrecognizedResourceType",type);
                (void) SetMagickResourceLimit((ResourceType) resource,limit);
                break;
              }
            break;
          }
        if (LocaleCompare("list",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                switch (*argv[i])
                {
                  case 'C':
                  case 'c':
                  {
                    if (LocaleCompare("Coder",argv[i]) == 0)
                      {
                        (void) ListCoderInfo((FILE *) NULL,exception);
                        break;
                      }
                    if (LocaleCompare("Color",argv[i]) == 0)
                      {
                        (void) ListColorInfo((FILE *) NULL,exception);
                        break;
                      }
                    if (LocaleCompare("Configure",argv[i]) == 0)
                      {
                        (void) ListConfigureInfo((FILE *) NULL,exception);
                        break;
                      }
                    ThrowMogrifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                  }
                  case 'D':
                  case 'd':
                  {
                    if (LocaleCompare("Delegate",argv[i]) == 0)
                      {
                        (void) ListDelegateInfo((FILE *) NULL,exception);
                        break;
                      }
                    ThrowMogrifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                  }
                  case 'F':
                  case 'f':
                  {
                    if (LocaleCompare("Format",argv[i]) == 0)
                      {
                        (void) ListMagickInfo((FILE *) NULL,exception);
                        break;
                      }
                    ThrowMogrifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                  }
                  case 'L':
                  case 'l':
                  {
                    if (LocaleCompare("Locale",argv[i]) == 0)
                      {
                        (void) ListLocaleInfo((FILE *) NULL,exception);
                        break;
                      }
                    if (LocaleCompare("Log",argv[i]) == 0)
                      {
                        (void) ListLogInfo((FILE *) NULL,exception);
                        break;
                      }
                    ThrowMogrifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                  }
                  case 'M':
                  case 'm':
                  {
                    if (LocaleCompare("Magic",argv[i]) == 0)
                      {
                        (void) ListMagicInfo((FILE *) NULL,exception);
                        break;
                      }
#if defined(SupportMagickModules)
                    if (LocaleCompare("Module",argv[i]) == 0)
                      {
                        (void) ListModuleInfo((FILE *) NULL,exception);
                        break;
                      }
#endif
                    ThrowMogrifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                  }
                  case 'R':
                  case 'r':
                  {
                    if (LocaleCompare("Resource",argv[i]) == 0)
                      {
                        (void) ListMagickResourceInfo((FILE *) NULL,exception);
                        break;
                      }
                    ThrowMogrifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                  }
                  case 'T':
                  case 't':
                  {
                    if (LocaleCompare("Type",argv[i]) == 0)
                      {
                        (void) ListTypeInfo((FILE *) NULL,exception);
                        break;
                      }
                    ThrowMogrifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                  }
                  default:
                    ThrowMogrifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                }
                return(MagickTrue);
              }
            break;
          }
        if (LocaleCompare("log",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if ((i == (long) argc) || (strchr(argv[i],'%') == (char *) NULL))
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                (void) SetLogFormat(argv[i]);
              }
            break;
          }
        if (LocaleCompare("loop",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'm':
      {
        if (LocaleCompare("map",option+1) == 0)
          {
            global_colormap=(MagickBooleanType) (*option == '+');
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
              }
            break;
          }
        if (LocaleCompare("mask",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
              }
            break;
          }
        if (LocaleCompare("matte",option+1) == 0)
          break;
        if (LocaleCompare("mattecolor",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                (void) QueryColorDatabase(argv[i],&image_info->matte_color,
                  exception);
              }
            break;
          }
        if (LocaleCompare("modulate",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("median",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("monochrome",option+1) == 0)
          {
            image_info->monochrome=(MagickBooleanType) (*option == '-');
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'n':
      {
        if (LocaleCompare("negate",option+1) == 0)
          break;
        if (LocaleCompare("noise",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            if (*option == '+')
              {
                long
                  noise;

                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                noise=ParseMagickOption(MagickNoiseOptions,MagickFalse,argv[i]);
                if (noise < 0)
                  ThrowMogrifyException(OptionError,"UnrecognizedNoiseType",
                    argv[i]);
              }
            break;
          }
        if (LocaleCompare("noop",option+1) == 0)
          break;
        if (LocaleCompare("normalize",option+1) == 0)
          break;
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'o':
      {
        if (LocaleCompare("opaque",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
              }
            break;
          }
        if (LocaleCompare("ordered-dither",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("orient",option+1) == 0)
          {
            long
              orientation;

            orientation=UndefinedOrientation;
            if (*option == '-')
              {
                i++;
                if (i == (long) (argc-1))
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                orientation=ParseMagickOption(MagickOrientationOptions,MagickFalse,
                  argv[i]);
                if (orientation < 0)
                  ThrowMogrifyException(OptionError,
                    "UnrecognizedImageOrientation",argv[i]);
                break;
              }
            image_info->orientation=(OrientationType) orientation;
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'p':
      {
        if (LocaleCompare("page",option+1) == 0)
          {
            (void) CloneString(&image_info->page,(char *) NULL);
            if (*option == '+')
              {
                char
                  *page;

                page=RemoveImageOption(image_info,argv[i]);
                if (page != (char *) NULL)
                  page=(char *) RelinquishMagickMemory(page);
                break;
              }
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            status=SetImageOption(image_info,"page",argv[i]);
            if (status == MagickFalse)
              ThrowMogrifyException(OptionError,"UnrecognizedOption",argv[i]);
            image_info->page=GetPageGeometry(argv[i]);
            break;
          }
        if (LocaleCompare("paint",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("pointsize",option+1) == 0)
          {
            image_info->pointsize=12;
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
                image_info->pointsize=atof(argv[i]);
              }
            break;
          }
        if (LocaleCompare("profile",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'q':
      {
        if (LocaleCompare("quality",option+1) == 0)
          {
            image_info->quality=UndefinedCompressionQuality;
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
                image_info->quality=(unsigned long) atol(argv[i]);
              }
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'r':
      {
        if (LocaleCompare("radial-blur",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            if (IsGeometry(argv[i]) == MagickFalse)
              ThrowMogrifyInvalidArgumentException(option,argv[i]);
            break;
          }
        if (LocaleCompare("raise",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            if (IsGeometry(argv[i]) == MagickFalse)
              ThrowMogrifyInvalidArgumentException(option,argv[i]);
            break;
          }
        if (LocaleCompare("random-threshold",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("red-primary",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("region",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("render",option+1) == 0)
          break;
        if (LocaleCompare("resample",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("resize",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("roll",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("rotate",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            if (IsGeometry(argv[i]) == MagickFalse)
              ThrowMogrifyInvalidArgumentException(option,argv[i]);
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 's':
      {
        if (LocaleCompare("sample",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("sampling-factor",option+1) == 0)
          {
            (void) CloneString(&image_info->sampling_factor,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
                (void) CloneString(&image_info->sampling_factor,argv[i]);
              }
            break;
          }
        if (LocaleCompare("scale",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("scene",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("seed",option+1) == 0)
          {
            unsigned long
              seed;

            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            seed=(unsigned long) atol(argv[i]);
            DistillRandomEvent((unsigned char *) &seed,sizeof(unsigned long));
            break;
          }
        if (LocaleCompare("segment",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("separate",option+1) == 0)
          break;
        if (LocaleCompare("shade",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            if (IsGeometry(argv[i]) == MagickFalse)
              ThrowMogrifyInvalidArgumentException(option,argv[i]);
            break;
          }
        if (LocaleCompare("sharpen",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            if (IsGeometry(argv[i]) == MagickFalse)
              ThrowMogrifyInvalidArgumentException(option,argv[i]);
            break;
          }
        if (LocaleCompare("shave",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("shear",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            if (IsGeometry(argv[i]) == MagickFalse)
              ThrowMogrifyInvalidArgumentException(option,argv[i]);
            break;
          }
        if (LocaleCompare("size",option+1) == 0)
          {
            (void) CloneString(&image_info->size,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
                (void) CloneString(&image_info->size,argv[i]);
              }
            break;
          }
        if (LocaleCompare("solarize",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("spread",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("stretch",option+1) == 0)
          {
            if (*option == '-')
              {
                long
                  stretch;

                i++;
                if (i == (long) (argc-1))
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                stretch=ParseMagickOption(MagickStretchOptions,MagickFalse,argv[i]);
                if (stretch < 0)
                  ThrowMogrifyException(OptionError,"UnrecognizedStyleType",
                    argv[i]);
                break;
              }
            break;
          }
        if (LocaleCompare("strip",option+1) == 0)
          break;
        if (LocaleCompare("stroke",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
              }
            break;
          }
        if (LocaleCompare("strokewidth",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("style",option+1) == 0)
          {
            if (*option == '-')
              {
                long
                  style;

                i++;
                if (i == (long) (argc-1))
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                style=ParseMagickOption(MagickStyleOptions,MagickFalse,argv[i]);
                if (style < 0)
                  ThrowMogrifyException(OptionError,"UnrecognizedStyleType",
                    argv[i]);
                break;
              }
            break;
          }
        if (LocaleCompare("support",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("swirl",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 't':
      {
        if (LocaleCompare("texture",option+1) == 0)
          {
            (void) CloneString(&image_info->texture,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                (void) CloneString(&image_info->texture,argv[i]);
              }
            break;
          }
        if (LocaleCompare("threshold",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("thumbnail",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("transparent",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
              }
            break;
          }
        if (LocaleCompare("treedepth",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("trim",option+1) == 0)
          break;
        if (LocaleCompare("type",option+1) == 0)
          {
            image_info->type=UndefinedType;
            if (*option == '-')
              {
                long
                  type;

                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                type=ParseMagickOption(MagickImageOptions,MagickFalse,argv[i]);
                if (type < 0)
                  ThrowMogrifyException(OptionError,"UnrecognizedImageType",
                    argv[i]);
                image_info->type=(ImageType) type;
              }
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'u':
      {
        if (LocaleCompare("undercolor",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
              }
            break;
          }
        if (LocaleCompare("units",option+1) == 0)
          {
            image_info->units=UndefinedResolution;
            if (*option == '-')
              {
                long
                  units;

                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                units=ParseMagickOption(MagickResolutionOptions,MagickFalse,argv[i]);
                if (units < 0)
                  ThrowMogrifyException(OptionError,"UnrecognizedUnitsType",
                    argv[i]);
              }
            break;
          }
        if (LocaleCompare("unsharp",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            if (IsGeometry(argv[i]) == MagickFalse)
              ThrowMogrifyInvalidArgumentException(option,argv[i]);
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'v':
      {
        if (LocaleCompare("verbose",option+1) == 0)
          {
            image_info->verbose=(MagickBooleanType) (*option == '-');
            break;
          }
        if (LocaleCompare("version",option+1) == 0)
          break;
        if (LocaleCompare("view",option+1) == 0)
          {
            (void) CloneString(&image_info->view,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                (void) CloneString(&image_info->view,argv[i]);
              }
            break;
          }
        if (LocaleCompare("virtual-pixel",option+1) == 0)
          {
            if (*option == '-')
              {
                long
                  method;

                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                method=ParseMagickOption(MagickVirtualPixelOptions,MagickFalse,
                  argv[i]);
                if (method < 0)
                  ThrowMogrifyException(OptionError,
                    "UnrecognizedVirtualPixelMethod",argv[i]);
              }
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'w':
      {
        if (LocaleCompare("wave",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowMogrifyException(OptionError,"MissingArgument",option);
            if (IsGeometry(argv[i]) == MagickFalse)
              ThrowMogrifyInvalidArgumentException(option,argv[i]);
            break;
          }
        if (LocaleCompare("weight",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) (argc-1))
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
              }
            break;
          }
        if (LocaleCompare("white-point",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("white-threshold",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowMogrifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowMogrifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
      }
      case '?':
        break;
      default:
        ThrowMogrifyException(OptionError,"UnrecognizedOption",option)
    }
  }
  if (i != argc)
    ThrowMogrifyException(OptionError,"MissingAnImageFilename",argv[i]);
  if (image == (Image *) NULL)
    ThrowMogrifyException(OptionError,"MissingAnImageFilename",argv[argc-1]);
  DestroyMogrify();
  return(status);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+     M o g r i f y I m a g e L i s t                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  MogrifyImageList() applies any command line options that might affect the
%  entire image list (e.g. -append, -coalesce, etc.).
%
%  The format of the MogrifyImage method is:
%
%      MagickBooleanType MogrifyImageList(const ImageInfo *image_info,
%        const int argc,char **argv,Image **images)
%
%  A description of each parameter follows:
%
%    o image_info: The image info..
%
%    o argc: Specifies a pointer to an integer describing the number of
%      elements in the argument vector.
%
%    o argv: Specifies a pointer to a text array containing the command line
%      arguments.
%
%    o images: The images.
%
%
*/
MagickExport MagickBooleanType MogrifyImageList(const ImageInfo *image_info,
  const int argc,char **argv,Image **images)
{
  ChannelType
    channel;

  char
    *option;

  ImageInfo
    *mogrify_info;

  long
    index;

  MagickBooleanType
    status;

  register long
    i;

  /*
    Apply options to the image list.
  */
  assert(image_info != (ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  assert(images != (Image **) NULL);
  assert((*images)->signature == MagickSignature);
  if ((*images)->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),(*images)->filename);
  if ((argc <= 0) || (*argv == (char *) NULL))
    return(MagickTrue);
  channel=(ChannelType) ((long) AllChannels &~ (long) OpacityChannel);
  status=MagickTrue;
  for (i=0; i < (long) argc; i++)
  {
    option=argv[i];
    if ((strlen(option) == 1) || ((*option != '-') && (*option != '+')))
      continue;
    switch (*(option+1))
    {
      case 'a':
      {
        if (LocaleCompare("append",option+1) == 0)
          {
            Image
              *append_image;

            (*images)->background_color=image_info->background_color;
            append_image=AppendImages(*images,(MagickBooleanType)
              (*option == '-'),&(*images)->exception);
            if (append_image == (Image *) NULL)
              break;
            InheritException(&append_image->exception,&(*images)->exception);
            DestroyImageList(*images);
            *images=append_image;
            break;
          }
        if (LocaleCompare("average",option+1) == 0)
          {
            Image
              *average_image;

            average_image=AverageImages(*images,&(*images)->exception);
            if (average_image == (Image *) NULL)
              break;
            InheritException(&average_image->exception,&(*images)->exception);
            DestroyImageList(*images);
            *images=average_image;
            break;
          }
        break;
      }
      case 'c':
      {
        if (LocaleCompare("channel",option+1) == 0)
          {
            if (*option == '+')
              {
                channel=(ChannelType) ((long) AllChannels &~
                  (long) OpacityChannel);
                break;
              }
            channel=(ChannelType) ParseChannelOption(argv[++i]);
            break;
          }
        if (LocaleCompare("coalesce",option+1) == 0)
          {
            Image
              *coalesce_image;

            coalesce_image=CoalesceImages(*images,&(*images)->exception);
            if (coalesce_image == (Image *) NULL)
              break;
            InheritException(&coalesce_image->exception,&(*images)->exception);
            DestroyImageList(*images);
            *images=coalesce_image;
            break;
          }
        if (LocaleCompare("combine",option+1) == 0)
          {
            Image
              *combine_image;

            combine_image=CombineImages(*images,channel,&(*images)->exception);
            if (combine_image == (Image *) NULL)
              break;
            InheritException(&combine_image->exception,&(*images)->exception);
            DestroyImageList(*images);
            *images=combine_image;
            break;
          }
        if (LocaleCompare("compose",option+1) == 0)
          {
            char
              composite_geometry[MaxTextExtent];

            CompositeOperator
              compose;

            Image
              *composite_image,
              *image,
              *mask_image;

            RectangleInfo
              geometry;

            compose=(CompositeOperator)
              ParseMagickOption(MagickCompositeOptions,MagickFalse,argv[++i]);
            image=(*images);
            composite_image=(*images)->next;
            if (composite_image == (Image *) NULL)
              break;
            mask_image=composite_image->next;
            image=CloneImage(image,0,0,MagickTrue,&(*images)->exception);
            if (image == (Image *) NULL)
              break;
            SetGeometry(composite_image,&geometry);
            (void) ParseAbsoluteGeometry(image->geometry,&geometry);
            (void) FormatMagickString(composite_geometry,MaxTextExtent,
              "%lux%lu%+ld%+ld",composite_image->columns,composite_image->rows,
              geometry.x,geometry.y);
            (void) ParseGravityGeometry(image,composite_geometry,&geometry);
            if (mask_image != (Image *) NULL)
              {
                SetImageType(composite_image,TrueColorMatteType);
                if (composite_image->matte == MagickFalse)
                  SetImageOpacity(composite_image,OpaqueOpacity);
                status&=CompositeImage(composite_image,CopyOpacityCompositeOp,
                  mask_image,0,0);
              }
            (void) CompositeImage(image,compose,composite_image,geometry.x,
              geometry.y);
            InheritException(&image->exception,&(*images)->exception);
            DestroyImageList(*images);
            *images=image;
            break;
          }
        break;
      }
      case 'd':
      {
        if (LocaleCompare("deconstruct",option+1) == 0)
          {
            Image
              *deconstruct_image;

            deconstruct_image=DeconstructImages(*images,&(*images)->exception);
            if (deconstruct_image == (Image *) NULL)
              break;
            InheritException(&deconstruct_image->exception,
              &(*images)->exception);
            DestroyImageList(*images);
            *images=deconstruct_image;
            break;
          }
        if (LocaleCompare("delete",option+1) == 0)
          {
            Image
              *delete_image;

            index=(-1);
            if (*option != '+')
              index=atol(argv[++i]);
            delete_image=GetImageFromList(*images,index);
            if (delete_image == (Image *) NULL)
              {
                (void) ThrowMagickException(&(*images)->exception,
                  GetMagickModule(),OptionError,"NoSuchImage",argv[i]);
                break;
              }
            DeleteImageFromList(&delete_image);
            *images=GetFirstImageInList(delete_image);
            break;
          }
        break;
      }
      case 'f':
      {
        if (LocaleCompare("flatten",option+1) == 0)
          {
            Image
              *flatten_image;

            flatten_image=FlattenImages(*images,&(*images)->exception);
            if (flatten_image == (Image *) NULL)
              break;
            InheritException(&flatten_image->exception,&(*images)->exception);
            DestroyImageList(*images);
            *images=flatten_image;
            break;
          }
        if (LocaleCompare("fx",option+1) == 0)
          {
            Image
              *fx_image;

            fx_image=FxImageChannel(*images,channel,argv[++i],
              &(*images)->exception);
            if (fx_image == (Image *) NULL)
              break;
            InheritException(&fx_image->exception,&(*images)->exception);
            DestroyImageList(*images);
            *images=fx_image;
            break;
          }
        break;
      }
      case 'i':
      {
        if (LocaleCompare("insert",option+1) == 0)
          {
            Image
              *p,
              *q;

            index=0;
            if (*option != '+')
              index=atol(argv[++i]);
            p=RemoveLastImageFromList(images);
            if (p == (Image *) NULL)
              {
                (void) ThrowMagickException(&(*images)->exception,
                  GetMagickModule(),OptionError,"NoSuchImage",argv[i]);
                break;
              }
            q=p;
            if (index == 0)
              PrependImageToList(&q,*images);
            else
              {
                 q=GetImageFromList(*images,index-1);
                 if (q == (Image *) NULL)
                   {
                     (void) ThrowMagickException(&(*images)->exception,
                       GetMagickModule(),OptionError,"NoSuchImage",argv[i]);
                     break;
                   }
                InsertImageInList(&q,p);
              }
            *images=GetFirstImageInList(q);
            break;
          }
        break;
      }
      case 'm':
      {
        if (LocaleCompare("map",option+1) == 0)
          {
             if (*option == '+')
               {
                 (void) MapImages(*images,(Image *) NULL,image_info->dither);
                 break;
               }
             i++;
             break;
          }
        if (LocaleCompare("morph",option+1) == 0)
          {
            Image
              *morph_image;

            morph_image=MorphImages(*images,(unsigned long) atol(argv[++i]),
              &(*images)->exception);
            if (morph_image == (Image *) NULL)
              break;
            InheritException(&morph_image->exception,&(*images)->exception);
            DestroyImageList(*images);
            *images=morph_image;
            break;
          }
        if (LocaleCompare("mosaic",option+1) == 0)
          {
            Image
              *mosaic_image;

            mosaic_image=MosaicImages(*images,&(*images)->exception);
            if (mosaic_image == (Image *) NULL)
              break;
            InheritException(&mosaic_image->exception,&(*images)->exception);
            DestroyImageList(*images);
            *images=mosaic_image;
            break;
          }
        break;
      }
      case 'p':
      {
        if (LocaleCompare("process",option+1) == 0)
          {
            char
              *arguments,
              breaker,
              quote,
              *token;

            int
              next,
              status;

            size_t
              length;

            TokenInfo
              token_info;

            length=strlen(argv[++i]);
            token=(char *) AcquireMagickMemory(length+1);
            if (token == (char *) NULL)
              break;
            next=0;
            arguments=argv[i];
            status=Tokenizer(&token_info,0,token,length,arguments,(char *) "",
              (char *) "=",(char *) "\"",'\0',&breaker,&next,&quote);
            if (status == 0)
              {
                char
                  *argv;

                argv=&(arguments[next]);
                (void) ExecuteModuleProcess(token,&(*images),1,&argv);
              }
            token=(char *) RelinquishMagickMemory(token);
            break;
          }
        break;
      }
      case 's':
      {
        if (LocaleCompare("swap",option+1) == 0)
          {
            Image
              *p,
              *q,
              *swap;

            long
              swap_index;

            index=(-1);
            swap_index=(-2);
            if (*option != '+')
              {
                GeometryInfo
                  geometry_info;

                MagickStatusType
                  flags;

                flags=ParseGeometry(argv[++i],&geometry_info);
                index=(long) geometry_info.rho;
                if ((flags & SigmaValue) != 0)
                  swap_index=(long) geometry_info.sigma;
              }
            p=GetImageFromList(*images,index);
            q=GetImageFromList(*images,swap_index);
            if ((p == (Image *) NULL) || (q == (Image *) NULL))
              {
                (void) ThrowMagickException(&(*images)->exception,
                  GetMagickModule(),OptionError,"NoSuchImage",argv[i]);
                break;
              }
            swap=CloneImage(p,0,0,MagickTrue,&(*images)->exception);
            ReplaceImageInList(&p,CloneImage(q,0,0,MagickTrue,&(*images)->exception));
            ReplaceImageInList(&q,swap);
            *images=GetFirstImageInList(q);
            break;
          }
        break;
      }
      case 'w':
      {
        if (LocaleCompare("write",option+1) == 0)
          {
            Image
              *write_images;

            write_images=(*images);
            if (*option == '+')
              write_images=CloneImageList(*images,&(*images)->exception);
            mogrify_info=CloneImageInfo(image_info);
            status&=WriteImages(mogrify_info,write_images,argv[++i],
              &(*images)->exception);
            DestroyImageInfo(mogrify_info);
            if (*option == '+')
              DestroyImageList(write_images);
            break;
          }
        break;
      }
      default:
        break;
    }
  }
  return(status);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
+     M o g r i f y I m a g e s                                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  MogrifyImages() applies image processing options to a sequence of images as
%  prescribed by command line options.
%
%  The format of the MogrifyImage method is:
%
%      MagickBooleanType MogrifyImages(const ImageInfo *image_info,
%        const int argc,char **argv,Image **images)
%
%  A description of each parameter follows:
%
%    o image_info: The image info..
%
%    o argc: Specifies a pointer to an integer describing the number of
%      elements in the argument vector.
%
%    o argv: Specifies a pointer to a text array containing the command line
%      arguments.
%
%    o images: The images.
%
%
*/
MagickExport MagickBooleanType MogrifyImages(const ImageInfo *image_info,
  const int argc,char **argv,Image **images)
{
#define MogrifyImageTag  "Mogrify/Image"

  char
    *option;

  Image
    *image,
    *mogrify_images;

  MagickBooleanType
    status;

  MonitorHandler
    handler;

  register long
    i;

  unsigned long
    number_images,
    scene;

  /*
    Apply options to individual images in the list.
  */
  assert(image_info != (ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  assert(images != (Image **) NULL);
  assert((*images)->signature == MagickSignature);
  if ((*images)->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),(*images)->filename);
  if ((argc <= 0) || (*argv == (char *) NULL))
    return(MagickTrue);
  scene=MagickFalse;
  for (i=0; i < (long) argc; i++)
  {
    option=argv[i];
    if ((strlen(option) <= 1) || ((*option != '-') && (*option != '+')))
      continue;
    switch (*(option+1))
    {
      case 's':
      {
        if (LocaleCompare("scene",option+1) == 0)
          scene=MagickTrue;
        break;
      }
      default:
        break;
    }
  }
  status=MagickTrue;
  mogrify_images=NewImageList();
  number_images=GetImageListLength(*images);
  for (i=0; i < (long) number_images; i++)
  {
    image=RemoveFirstImageFromList(images);
    handler=SetMonitorHandler((MonitorHandler) NULL);
    status&=MogrifyImage(image_info,argc,argv,&image);
    (void) SetMonitorHandler(handler);
    if (scene != MagickFalse)
      image->scene=(unsigned long) i;
    if (image_info->verbose != MagickFalse)
      (void) DescribeImage(image,stdout,MagickFalse);
    AppendImageToList(&mogrify_images,image);
    status=MagickMonitor(MogrifyImageTag,i,number_images,&image->exception);
    if (status == MagickFalse)
      break;
  }
  status&=MogrifyImageList(image_info,argc,argv,&mogrify_images);
  *images=mogrify_images;
  return(status);
}
