/*  gxfce
 *  Copyright (C) 2003 Olivier Fourdan (fourdan@xfce.org)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef GDK_MULTIHEAD_SAFE
#undef GDK_MULTIHEAD_SAFE
#endif

#ifdef USE_GDK_SCREEN
#undef USE_GDK_SCREEN
#endif

#include <glib.h>

#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#ifdef HAVE_LIBXINERAMA
#include <X11/extensions/Xinerama.h>
#endif

#include "xinerama.h"

#ifdef EMULATE_XINERAMA
/* NOTE : Look in xinerama.h to define EMULATE_XINERAMA
 * (do not do it here)
 * 
 * We simulate 2 screens side by side on a 1280x1024 screen :
 * Left screen is 640x600 pixels wide with 212 unused pixels 
 *   on top and bottom of screen (this is to simulate the
 *   black holes as show below :
 * Right screen is 640x1024 pixels wide
 *
 *  //////////////////+-----------------+
 *  ////BLACK HOLE////|                 |
 *  //////////////////|                 |
 *  +-----------------|                 |
 *  |                 |                 |
 *  |                 |                 |
 *  |      First      |      Second     |
 *  |     Screen      |      Screen     |
 *  |                 |                 |
 *  |                 |                 |
 *  +-----------------|                 |
 *  //////////////////|                 |
 *  ////BLACK HOLE////|                 |
 *  //////////////////+-----------------+
 *
 */

static XineramaScreenInfo emulate_infos[2] = {
    {0, 0, 212, 640, 600},
    {1, 640, 0, 640, 1024}
};
static int emulate_heads = 2;
#endif

#if defined(HAVE_LIBXINERAMA) || defined(EMULATE_XINERAMA)
static XineramaScreenInfo *xinerama_infos;
static int xinerama_heads;
#endif

static gboolean enable_xinerama = FALSE;

#if defined(HAVE_LIBXINERAMA)
int xinerama_major, xinerama_minor;
#endif

gboolean
xineramaInit (Display * dpy)
{
#ifdef EMULATE_XINERAMA
    xinerama_infos = emulate_infos;
    xinerama_heads = emulate_heads;
    enable_xinerama = TRUE;
#else /* EMULATE_XINERAMA */
#ifdef HAVE_LIBXINERAMA
    if (!XineramaQueryExtension (dpy, &xinerama_major, &xinerama_minor))
    {
        xinerama_heads = 0;
        enable_xinerama = FALSE;
    }
    else
    {
        xinerama_infos = XineramaQueryScreens (dpy, &xinerama_heads);
        enable_xinerama = TRUE;
    }
#else /* HAVE_LIBXINERAMA */
    enable_xinerama =  FALSE;
#endif /* HAVE_LIBXINERAMA */
#endif /* EMULATE_XINERAMA */

    return enable_xinerama;
}

void
xineramaFree (void)
{
#ifndef EMULATE_XINERAMA
#ifdef HAVE_LIBXINERAMA
    XFree (xinerama_infos);
#endif /* HAVE_LIBXINERAMA */
#endif /* EMULATE_XINERAMA */
}

int
xineramaGetHeads (void)
{
#if defined(HAVE_LIBXINERAMA) || defined(EMULATE_XINERAMA)
    return xinerama_heads;
#else /* HAVE_LIBXINERAMA */
    return 0;
#endif /* HAVE_LIBXINERAMA */
}

/* Xinerama handling routines */

#if defined(HAVE_LIBXINERAMA) || defined(EMULATE_XINERAMA)
static int
findhead (int x, int y)
{
    extern XineramaScreenInfo *xinerama_infos;
    extern int xinerama_heads;
    extern gboolean enable_xinerama;
    static int cache_x;
    static int cache_y;
    static int cache_head;
    static gboolean cache_init = FALSE;

    int head, closest_head;
    int dx, dy;
    int center_x, center_y;
    int distsquare, min_distsquare;

    /* Cache system */
    if ((cache_init) && (x == cache_x) && (y == cache_y))
        return (cache_head);

    /* Okay, now we consider the cache has been initialized */
    cache_init = TRUE;
    cache_x = x;
    cache_y = y;

    if ((xinerama_heads == 0) || (xinerama_infos == NULL)
        || (!enable_xinerama))
        /* Xinerama extensions are disabled */
    {
        cache_head = 0;
        return (0);
    }

    for (head = 0; head < xinerama_heads; head++)
    {
        if ((xinerama_infos[head].x_org <= x)
            && ((xinerama_infos[head].x_org + xinerama_infos[head].width) > x)
            && (xinerama_infos[head].y_org <= y)
            && ((xinerama_infos[head].y_org + xinerama_infos[head].height) >
                y))
        {
            cache_head = head;
            return (head);
        }
    }
    /* No head has been eligible, use the closest one */

    center_x = xinerama_infos[0].x_org + (xinerama_infos[0].width / 2);
    center_y = xinerama_infos[0].y_org + (xinerama_infos[0].height / 2);

    dx = x - center_x;
    dy = y - center_y;

    min_distsquare = (dx * dx) + (dy * dy);
    closest_head = 0;

    for (head = 1; head < xinerama_heads; head++)
    {
        center_x =
            xinerama_infos[head].x_org + (xinerama_infos[head].width / 2);
        center_y =
            xinerama_infos[head].y_org + (xinerama_infos[head].height / 2);

        dx = x - center_x;
        dy = y - center_y;

        distsquare = (dx * dx) + (dy * dy);

        if (distsquare < min_distsquare)
        {
            min_distsquare = distsquare;
            closest_head = head;
        }
    }
    cache_head = closest_head;
    return (closest_head);
}
#endif

int
MyDisplayFullHeight (Display * dpy, int screen)
{

#ifdef USE_GDK_SCREEN
    return gdk_screen_height ();
#else /* USE_GDK_SCREEN */
    return XDisplayHeight (dpy, screen);
#endif /* USE_GDK_SCREEN */
}

int
MyDisplayFullWidth (Display * dpy, int screen)
{
#ifdef USE_GDK_SCREEN
    return gdk_screen_width ();
#else /* USE_GDK_SCREEN */
    return XDisplayWidth (dpy, screen);
#endif /* USE_GDK_SCREEN */
}

int
MyDisplayHeight (Display * dpy, int screen, int x, int y)
{
#if defined(HAVE_LIBXINERAMA) || defined(EMULATE_XINERAMA)
    int head;

    if ((xinerama_heads == 0) || (xinerama_infos == NULL)
        || (!enable_xinerama))
    {
        /* Xinerama extensions are disabled */
        return MyDisplayFullHeight (dpy, screen);
    }
    head = findhead (x, y);
    return xinerama_infos[head].height;
#else
    return MyDisplayFullHeight (dpy, screen);
#endif /* HAVE_LIBXINERAMA */
}

int
MyDisplayWidth (Display * dpy, int screen, int x, int y)
{
#if defined(HAVE_LIBXINERAMA) || defined(EMULATE_XINERAMA)
    int head;

    if ((xinerama_heads == 0) || (xinerama_infos == NULL)
        || (!enable_xinerama))
    {
        /* Xinerama extensions are disabled */
        return MyDisplayFullWidth (dpy, screen);
    }
    head = findhead (x, y);
    return (xinerama_infos[head].width);
#else /* HAVE_LIBXINERAMA */
    return MyDisplayFullWidth (dpy, screen);
#endif /* HAVE_LIBXINERAMA */
}

int
MyDisplayX (int x, int y)
{
#if defined(HAVE_LIBXINERAMA) || defined(EMULATE_XINERAMA)
    int head;

    if ((xinerama_heads == 0) || (xinerama_infos == NULL)
        || (!enable_xinerama))
        /* Xinerama extensions are disabled */
        return (0);

    head = findhead (x, y);
    return (xinerama_infos[head].x_org);
#else /* HAVE_LIBXINERAMA */
    return (0);
#endif /* HAVE_LIBXINERAMA */
}


int
MyDisplayY (int x, int y)
{
#if defined(HAVE_LIBXINERAMA) || defined(EMULATE_XINERAMA)
    int head;

    if ((xinerama_heads == 0) || (xinerama_infos == NULL)
        || (!enable_xinerama))
        /* Xinerama extensions are disabled */
        return (0);

    head = findhead (x, y);
    return (xinerama_infos[head].y_org);
#else /* HAVE_LIBXINERAMA */
    return (0);
#endif /* HAVE_LIBXINERAMA */
}

int
MyDisplayMaxX (Display * dpy, int screen, int x, int y)
{
#if defined(HAVE_LIBXINERAMA) || defined(EMULATE_XINERAMA)
    int head;

    if ((xinerama_heads == 0) || (xinerama_infos == NULL)
        || (!enable_xinerama))
    {
        /* Xinerama extensions are disabled */
        return MyDisplayFullWidth (dpy, screen);
    }
    head = findhead (x, y);
    return (xinerama_infos[head].x_org + xinerama_infos[head].width);
#else /* HAVE_LIBXINERAMA */
    return MyDisplayFullWidth (dpy, screen);
#endif /* HAVE_LIBXINERAMA */
}

int
MyDisplayMaxY (Display * dpy, int screen, int x, int y)
{
#if defined(HAVE_LIBXINERAMA) || defined(EMULATE_XINERAMA)
    int head;

    if ((xinerama_heads == 0) || (xinerama_infos == NULL)
        || (!enable_xinerama))
    {
        /* Xinerama extensions are disabled */
        return MyDisplayFullHeight (dpy, screen);
    }

    head = findhead (x, y);
    return (xinerama_infos[head].y_org + xinerama_infos[head].height);
#else /* HAVE_LIBXINERAMA */
    return MyDisplayFullHeight (dpy, screen);
#endif /* HAVE_LIBXINERAMA */
}

gboolean
isRightMostHead (Display * dpy, int screen, int x, int y)
{
#if defined(HAVE_LIBXINERAMA) || defined(EMULATE_XINERAMA)
    int head;
    int i;
    
    if ((xinerama_heads == 0) || (xinerama_infos == NULL)
        || (!enable_xinerama))
    {
        return TRUE;
    }
    
    head = findhead (x, y);
    for (i = 0; i < xinerama_heads; i++)
    {
        if (head != i)
        {
            int y1, y2;
            
            y1 = MAX(xinerama_infos[i].y_org, xinerama_infos[head].y_org);
            y2 = MIN(xinerama_infos[i].y_org + xinerama_infos[i].height,
                     xinerama_infos[head].y_org + xinerama_infos[head].height);

            if ((xinerama_infos[i].x_org >= xinerama_infos[head].x_org + 
                 xinerama_infos[head].width) && (y2 - y1 > 0))
            {
                return FALSE;
            }
        }
    }
    return TRUE;
#else /* HAVE_LIBXINERAMA */
    return TRUE;
#endif /* HAVE_LIBXINERAMA */
}

gboolean
isLeftMostHead (Display * dpy, int screen, int x, int y)
{
#if defined(HAVE_LIBXINERAMA) || defined(EMULATE_XINERAMA)
    int head;
    int i;
    
    if ((xinerama_heads == 0) || (xinerama_infos == NULL)
        || (!enable_xinerama))
    {
        return TRUE;
    }

    head = findhead (x, y);
    for (i = 0; i < xinerama_heads; i++)
    {
        if (head != i)
        {
            int y1, y2;
            
            y1 = MAX(xinerama_infos[i].y_org, xinerama_infos[head].y_org);
            y2 = MIN(xinerama_infos[i].y_org + xinerama_infos[i].height,
                     xinerama_infos[head].y_org + xinerama_infos[head].height);

            if ((xinerama_infos[i].x_org + xinerama_infos[head].width 
                <= xinerama_infos[head].x_org) && (y2 - y1 > 0))
            {
                return FALSE;
            }
        }
    }
    return TRUE;
#else
    return TRUE;
#endif
}

gboolean
isTopMostHead (Display * dpy, int screen, int x, int y)
{
#if defined(HAVE_LIBXINERAMA) || defined(EMULATE_XINERAMA)
    int head;
    int i;
    
    if ((xinerama_heads == 0) || (xinerama_infos == NULL)
        || (!enable_xinerama))
    {
        return TRUE;
    }

    head = findhead (x, y);
    for (i = 0; i < xinerama_heads; i++)
    {
        if (head != i)
        {
            int x1, x2;
            
            x1 = MAX(xinerama_infos[i].x_org, xinerama_infos[head].x_org);
            x2 = MIN(xinerama_infos[i].x_org + xinerama_infos[i].width,
                     xinerama_infos[head].x_org + xinerama_infos[head].width);

            if ((xinerama_infos[i].y_org + xinerama_infos[head].height 
                <= xinerama_infos[head].y_org) && (x2 - x1 > 0))
            {
                return FALSE;
            }
        }
    }
    return TRUE;
#else
    return TRUE;
#endif
}

gboolean
isBottomMostHead (Display * dpy, int screen, int x, int y)
{
#if defined(HAVE_LIBXINERAMA) || defined(EMULATE_XINERAMA)
    int head;
    int i;
    
    if ((xinerama_heads == 0) || (xinerama_infos == NULL)
        || (!enable_xinerama))
    {
        return TRUE;
    }

    head = findhead (x, y);
    for (i = 0; i < xinerama_heads; i++)
    {
        if (head != i)
        {
            int x1, x2;
            
            x1 = MAX(xinerama_infos[i].x_org, xinerama_infos[head].x_org);
            x2 = MIN(xinerama_infos[i].x_org + xinerama_infos[i].width,
                     xinerama_infos[head].x_org + xinerama_infos[head].width);

            if ((xinerama_infos[i].y_org >= xinerama_infos[head].y_org + 
                 xinerama_infos[head].height) && (x2 - x1 > 0))
            {
                return FALSE;
            }
        }
    }
    return TRUE;
#else /* HAVE_LIBXINERAMA */
    return TRUE;
#endif /* HAVE_LIBXINERAMA */
}
