/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * Copyright (C) 2010 Red Hat, Inc.
 *
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <string.h>

#include "test-wmc-escaping.h"
#include "utils.h"

static const char data1[] = {
    0x70, 0x68, 0x6f, 0x6e, 0x65, 0x66, 0x61, 0x69, 0x6c, 0x75, 0x72, 0x65, 
    0x0a, 0x6e, 0x6f, 0x63, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x69, 0x6f, 
    0x6e, 0x74, 0x6f, 0x70, 0x68, 0x6f, 0x6e, 0x65, 0x0a, 0x70, 0x68, 0x6f, 
    0x6e, 0x7e, 0x7e, 0x7e, 0x7d, 0x7d, 0x7e, 0x7d, 0x7e, 0x7d, 0x7e, 0x6e, 
    0x6b, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x0a, 0x6f, 0x70, 
    0x65, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x6e, 0x6f, 0x74, 0x61, 0x6c, 
    0x6c, 0x6f, 0x77, 0x65, 0x64, 0x0a, 0x6f, 0x70, 0x65, 0x72, 0x61, 0x74, 
    0x69, 0x6f, 0x6e, 0x6e, 0x6f, 0x74, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 
    0x74, 0x65, 0x64, 0x0a, 0x70, 0x68, 0x73, 0x69, 0x6d, 0x70, 0x69, 0x6e, 
    0x72, 0x65, 0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 0x0a, 0x70, 0x68, 0x66, 
    0x73, 0x69, 0x6d, 0x70, 0x69, 0x6e, 0x72, 0x65, 0x71, 0x75, 0x69, 0x72, 
    0x65, 0x64, 0x0a, 0x70, 0x68, 0x66, 0x73, 0x69, 0x6d, 0x70, 0x75, 0x6b, 
    0x72, 0x65, 0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 0x0a, 0x73, 0x69, 0x6d, 
    0x6e, 0x6f, 0x74, 0x69, 0x6e, 0x73, 0x65, 0x72, 0x74, 0x65, 0x64, 0x0a, 
    0x73, 0x69, 0x6d, 0x70, 0x69, 0x6e, 0x72, 0x65, 0x71, 0x75, 0x69, 0x72, 
    0x65, 0x64, 0x0a, 0x73, 0x69, 0x6d, 0x70, 0x75, 0x6b, 0x72, 0x65, 0x71, 
    0x75, 0x69, 0x72, 0x65, 0x64, 0x0a, 0x73, 0x69, 0x6d, 0x66, 0x61, 0x69, 
    0x6c, 0x75, 0x72, 0x65, 0x0a, 0x73, 0x69, 0x6d, 0x62, 0x75, 0x73, 0x79, 
    0x0a, 0x73, 0x69, 0x6d, 0x77, 0x72, 0x6f, 0x6e, 0x67, 0x0a, 0x69, 0x6e, 
    0x63, 0x6f, 0x72, 0x72, 0x65, 0x63, 0x74, 0x70, 0x61, 0x73, 0x73, 0x77, 
    0x6f, 0x72, 0x64, 0x0a, 0x73, 0x69, 0x6d, 0x70, 0x69, 0x6e, 0x32, 0x72, 
    0x65, 0x71, 0x75, 0x69
};

static const char expected1[] = {
    0x70, 0x68, 0x6f, 0x6e, 0x65, 0x66, 0x61, 0x69, 0x6c, 0x75, 0x72, 0x65, 
    0x0a, 0x6e, 0x6f, 0x63, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x69, 0x6f, 
    0x6e, 0x74, 0x6f, 0x70, 0x68, 0x6f, 0x6e, 0x65, 0x0a, 0x70, 0x68, 0x6f, 
    0x6e, 0x7d, 0x5e, 0x7d, 0x5e, 0x7d, 0x5e, 0x7d, 0x5d, 0x7d, 0x5d, 0x7d, 
    0x5e, 0x7d, 0x5d, 0x7d, 0x5e, 0x7d, 0x5d, 0x7d, 0x5e, 0x6e, 0x6b, 0x72, 
    0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x0a, 0x6f, 0x70, 0x65, 0x72, 
    0x61, 0x74, 0x69, 0x6f, 0x6e, 0x6e, 0x6f, 0x74, 0x61, 0x6c, 0x6c, 0x6f, 
    0x77, 0x65, 0x64, 0x0a, 0x6f, 0x70, 0x65, 0x72, 0x61, 0x74, 0x69, 0x6f, 
    0x6e, 0x6e, 0x6f, 0x74, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 
    0x64, 0x0a, 0x70, 0x68, 0x73, 0x69, 0x6d, 0x70, 0x69, 0x6e, 0x72, 0x65, 
    0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 0x0a, 0x70, 0x68, 0x66, 0x73, 0x69, 
    0x6d, 0x70, 0x69, 0x6e, 0x72, 0x65, 0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 
    0x0a, 0x70, 0x68, 0x66, 0x73, 0x69, 0x6d, 0x70, 0x75, 0x6b, 0x72, 0x65, 
    0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 0x0a, 0x73, 0x69, 0x6d, 0x6e, 0x6f, 
    0x74, 0x69, 0x6e, 0x73, 0x65, 0x72, 0x74, 0x65, 0x64, 0x0a, 0x73, 0x69, 
    0x6d, 0x70, 0x69, 0x6e, 0x72, 0x65, 0x71, 0x75, 0x69, 0x72, 0x65, 0x64, 
    0x0a, 0x73, 0x69, 0x6d, 0x70, 0x75, 0x6b, 0x72, 0x65, 0x71, 0x75, 0x69, 
    0x72, 0x65, 0x64, 0x0a, 0x73, 0x69, 0x6d, 0x66, 0x61, 0x69, 0x6c, 0x75, 
    0x72, 0x65, 0x0a, 0x73, 0x69, 0x6d, 0x62, 0x75, 0x73, 0x79, 0x0a, 0x73, 
    0x69, 0x6d, 0x77, 0x72, 0x6f, 0x6e, 0x67, 0x0a, 0x69, 0x6e, 0x63, 0x6f, 
    0x72, 0x72, 0x65, 0x63, 0x74, 0x70, 0x61, 0x73, 0x73, 0x77, 0x6f, 0x72, 
    0x64, 0x0a, 0x73, 0x69, 0x6d, 0x70, 0x69, 0x6e, 0x32, 0x72, 0x65, 0x71, 
    0x75, 0x69
};

void
test_escape1 (void *f, void *data)
{
    char escaped[1024];
    size_t len;

    /* Ensure that escaping in general works */
    len = hdlc_escape (data1, sizeof (data1), FALSE, escaped, sizeof (escaped));
    g_assert (len == 266);
    g_assert (len == sizeof (expected1));
    g_assert (memcmp (escaped, expected1, len) == 0);
}

static const char data2[] = {
    0x4b, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x3f,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff
};

void
test_escape2 (void *f, void *data)
{
    char escaped[1024];
    size_t len;

    /* Ensure that escaping data that doesn't need escaping works */
    len = hdlc_escape (data2, sizeof (data2), FALSE, escaped, sizeof (escaped));
    g_assert (len == sizeof (data2));
    g_assert (memcmp (escaped, data2, len) == 0);
}

static const char data_ctrl_src[] = {
    0xc8, 0x0d, 0xda, 0x07, 0x0c, 0x00, 0x14, 0x00, 0x10, 0x00, 0x19, 0x00,
    0x04, 0x00, 0x01, 0x00, 0x07, 0x88
};

static const char data_ctrl_expected[] = {
    0xc8, 0x7d, 0x2d, 0xda, 0x7d, 0x27, 0x7d, 0x2c, 0x7d, 0x20, 0x7d, 0x34,
    0x7d, 0x20, 0x7d, 0x30, 0x7d, 0x20, 0x7d, 0x39, 0x7d, 0x20, 0x7d, 0x24,
    0x7d, 0x20, 0x7d, 0x21, 0x7d, 0x20, 0x7d, 0x27, 0x88
};

void
test_escape_ctrl (void *f, void *data)
{
    char escaped[1024];
    size_t len;

    len = hdlc_escape (data_ctrl_src, sizeof (data_ctrl_src), TRUE, escaped, sizeof (escaped));
    g_assert (len == sizeof (data_ctrl_expected));
    g_assert (memcmp (escaped, data_ctrl_expected, len) == 0);
}

void
test_escape_unescape (void *f, void *data)
{
    char escaped[512];
    char unescaped[512];
    size_t len, unlen;
    wmcbool escaping = FALSE;

    /* Ensure that escaping data that needs escaping, and then unescaping it,
     * produces the exact same data as was originally escaped.
     */
    len = hdlc_escape (data1, sizeof (data1), FALSE, escaped, sizeof (escaped));
    unlen = hdlc_unescape (escaped, len, unescaped, sizeof (unescaped), &escaping);

    g_assert (unlen == sizeof (data1));
    g_assert (memcmp (unescaped, data1, unlen) == 0);
}

void
test_escape_unescape_ctrl (void *f, void *data)
{
    char escaped[512];
    char unescaped[512];
    size_t len, unlen;
    wmcbool escaping = FALSE;

    /* Ensure that escaping data that needs escaping, and then unescaping it,
     * produces the exact same data as was originally escaped.
     */
    len = hdlc_escape (data_ctrl_src, sizeof (data_ctrl_src), TRUE, escaped, sizeof (escaped));
    g_assert (memcmp (escaped, data_ctrl_expected, len) == 0);

    unlen = hdlc_unescape (escaped, len, unescaped, sizeof (unescaped), &escaping);
    g_assert (unlen == sizeof (data_ctrl_src));
    g_assert (memcmp (unescaped, data_ctrl_src, unlen) == 0);
}

