/* Find file command implementation
   Copyright (C) 1994 Mauricio Plaza
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <ncurses.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <errno.h>
#include <dirent.h>
#include <unistd.h>

#include "dir.h"
#include "panel.h"
#include "input.h"
#include "global.h"
#include "dialog.h"
#include "menu.h"
#include "file.h"
#include "main.h"
#include "util.h"
#include "find.h"
#include "color.h"

static char rcsid [] = "$Header: /usr/users/miguel/c/CVS/nc/find.c,v 1.7 1994/09/08 19:28:02 miguel Exp $";

static Input *in;          /* Asks the file to search           */
static WINDOW *text_win;   /* Contains the Menu and Text        */
static WINDOW *file_win;   /* Contains the files found          */
static find_list *list;    /* double-linked list of files found */
static find_list *top_scr; /* Pointer to top of screen file     */
FStack *directories;       /* To search in all the SUB-DIR's    */
int    scr_pos;            /* Can't be static because it has to */
			   /* be changed each time the find menu */
			   /* it's called .*/

extern WINDOW *top_window;   /* top_widnow (see dialog.c for details)  */
extern Panel *current_panel; /* current_panel (see main.c for details) */
extern int mi_getch();       /* Alternate routine  (see key.c for details) */
extern void select_item();   /* See main.c for details */
extern void paint_dir();     /* See screen.c for details */
extern void destroy_dialog();   /* See dialog.c for details */
extern void wclr();			
/* Stack */

static void push_s (char *o)
{
    FStack *new;

    new = (FStack *) malloc (sizeof (FStack));
    new -> next = directories -> next;
    new -> dir_name = strdup (o);
    directories ->next = new;
}

static char *pop_s()
{
    FStack *aux;
    char *r;
    
    if (!directories || !directories->next)
	return 0;

    r = directories ->next->dir_name;
    aux = directories->next;
    directories->next = aux->next;
    return r;
}

/* Stack */

static char *input_find (char *def_text)
{
    int    c;
    char   *val;
    
    in = create_input (1, 3, text_win, INPUT_COLOR, MAX_FIND_COLS, def_text);

    while ((c = mi_getch ()) != '\n'){
	if (c == '\e')
	    return 0;
	handle_char (in, c);
    }
    wattrset (text_win, REVERSE_COLOR);
    val = in->buffer;
    destroy_input (in, IN_KEEP_BUFFER);
    return (val);
}

static void create_find_dialog()
{
    create_dialog (50, FIND_DIALOG_SIZE, " Find Window ","", 0);
    file_win = derwin (top_window, MAX_FIND_FLINES, MAX_FIND_COLS,  2, 3);
    text_win = derwin (top_window, MAX_FIND_TLINES, MAX_FIND_COLS, 13, 3);
    wattrset (text_win, REVERSE_COLOR);
    whline (text_win, ACS_HLINE, 0); 
    leaveok (text_win, TRUE);
    touchwin (text_win);
    touchwin (file_win);
    wrefresh (text_win);
    wrefresh (file_win);
    wrefresh (top_window);
}

static void init_find_menu (WINDOW *win, int param, char *text_menu[],
			    int Ncolor)
{
    int i;

    wattrset (win, Ncolor);
    for (i=0; i<param; i++)
	mvwprintw (win, 1, 1+i*12 ,*(text_menu+i) ) ;
    touchwin (win);
    wrefresh (win);
}

static void run_find_menu (WINDOW *win, int param, char *text_menu[],
			   int selected, int Scolor, int Ncolor)
{
    static int current_s = 1;

    wattrset (win, Ncolor);
    if (selected>param)
	selected = param;
    mvwprintw (win, 1, 1+(current_s-1)*12, *(text_menu+current_s-1));
    if (selected) {
	wattrset (win, Scolor);
	mvwprintw (win, 1, 1+(selected-1)*12, *(text_menu+selected-1));
	current_s = selected;
    }
    touchwin (win);
    wrefresh (win);
}

static int do_load_find_menu() {
    char   *menu_text[]={"   Find /  ","   Find ~  ",
			 "   Find .  ","    Quit   "};
    int    selection=1,key;

    init_find_menu (text_win, MAX_FIND_MENU, menu_text, REVERSE_COLOR );
    run_find_menu (text_win, MAX_FIND_MENU, menu_text, selection,
		   INPUT_COLOR, REVERSE_COLOR);
    
    while ((key=getch()) != '\n'){
	if (key=='\e')
	    return 0;
	if ( ((key == KEY_LEFT) || (key==XCTRL('b')))
	    && (selection>1))
	    selection--;
	if ( ((key == KEY_RIGHT) || (key==XCTRL('f')))
	    && (selection<MAX_FIND_MENU))
	    selection++;
	run_find_menu (text_win, MAX_FIND_MENU, menu_text, selection,
		       INPUT_COLOR, REVERSE_COLOR);
    }
    switch (selection){
    case 1: push_s ("/"); break;
    case 2: push_s (home_dir); break;
    case 3: push_s ("."); break;
    default:
	return 0;
    }
    return 1;
}

static char *trim2char (char *string, int lenght, int isdir)
{
    static char buf [100];
    int  res;

    if ((res = lenght - strlen (string)) > 0){
	if (isdir)
	    sprintf (buf, "%-48s", string);
	else
	    sprintf (buf, "\t%-40s", string);
    } else {
	buf [0] = 0;
    }
    return buf;
}

static void repaint_list (int NColor, int SColor) {
    int i;
    find_list *aux_list;

    wattrset (file_win, REVERSE_COLOR);
    scrollok (file_win, TRUE);
    while (list->ypos < top_scr->ypos) {
	wscrl (file_win, -1);
	top_scr = top_scr->up;
    }
    while (list-> ypos - top_scr-> ypos >= MAX_FIND_FLINES) {
	wscrl (file_win, 1);
	top_scr = top_scr->down;
    }
    scrollok (file_win, FALSE);
    aux_list = top_scr;
    for (i = 0; i < MAX_FIND_FLINES && aux_list; i++){
	if (aux_list-> selected)
	    wattrset (file_win, SColor );
	else
	    wattrset (file_win, NColor );
	mvwprintw (file_win, i, 0, "%s",
		   trim2char(aux_list->fname, MAX_FIND_COLS, aux_list->isdir));
	aux_list = aux_list->down;
    }
    touchwin (file_win);
    wrefresh (file_win);
}

static void put_on_list(char *name_file, char *directory, int isdir)
{
    find_list *new;
    int realpos;
    
    if (scr_pos > MAX_FIND_FLINES){ 
	scrollok (file_win, TRUE);
	wscrl (file_win, 1);
	scrollok (file_win, FALSE);
	realpos = MAX_FIND_FLINES-1;
    } else
	realpos = scr_pos;
    wattrset (file_win, REVERSE_COLOR);
    mvwprintw (file_win, realpos, 0, "%s",
	       trim2char(name_file, MAX_FIND_COLS, isdir));
    touchwin (file_win);
    wrefresh (file_win);
    
    /* list code */
    new = (find_list *) malloc (sizeof (find_list));
    new->ypos = scr_pos++;
    new->isdir = isdir;
    new->selected = 0;
    new->fname = strdup (name_file);
    new->path  = strdup (directory);
    if (list == NULL) {
	list = new;
	list->up   = NULL;
	list->down = NULL;
    } else {
	list->down = new; 
	new->up    = list;
	list       = new;
    }
}

static int add_list(char *pattern)
{
   static DIR    *dirp;       /* Pointer to Directory */
   struct dirent *dp;         /* Each File on SUB_DIR */
   struct stat   tmp_stat;    /* Status to check if it's a SUB_DIR */
   static char   *dir;        /* Name of the current dir */
   static char   old_dir [100];  /* Name of the last matching dir */
   static int    new_dir=1;   /* TRUE = pop (STACK) , Keep reading */
   char tmp_name [100];

   if (new_dir){
       if ( (dir = pop_s ()) ) {
	   new_dir = 0;
	   sprintf (old_dir, "\t");
	   dirp = opendir (dir);
       } else
	   return 0;
   }
   if (dirp == 0) {
       new_dir = 1;
       return 1;
   }
   if ( (dp = readdir (dirp)) ){
       if ( !strcmp(dp->d_name,".") || !strcmp(dp->d_name,".."))
	   return 1;
       sprintf (tmp_name, "%s/%s",dir,dp->d_name);
       lstat (tmp_name, &tmp_stat);
       wattrset (text_win, REVERSE_COLOR );
       mvwprintw ( text_win, 2, 1, "Searching on : %s",
		  trim2char(tmp_name, MAX_FIND_COLS-16, 1));
       touchwin (text_win);
       wrefresh (text_win);

       if (S_ISDIR(tmp_stat.st_mode))
	   push_s (tmp_name);

       if (regexp_match (pattern,dp->d_name)){
	   if (strcmp (dir, old_dir)) {
	       put_on_list (dir, dir, 1);
	       strcpy (old_dir,dir);
	   }
	   put_on_list (dp->d_name, dir, 0);
       }
   } else {
       new_dir = 1;
       closedir (dirp);
   }
   return 1;
}

static int select_and_change_panel()
{
    int  i;

    if ( !list ) return 0;
    if (chdir (list-> path) == -1)
	return 0;
    clean_dir (&current_panel->dir, current_panel->count);
    current_panel->count = do_load_dir (&current_panel->dir,
 					current_panel->sort_type,
					current_panel->reverse);
    current_panel->top_file = 0;
    current_panel->selected = 0;
    current_panel->marked = 0;
    current_panel->total = 0;
    getwd (current_panel->cwd);
    select_item (current_panel);
    paint_panel (current_panel);
    
    if (!list->isdir){
	for (i=0; strcmp (list->fname, current_panel->dir.list[i].fname); i++)
	    ;
	current_panel->selected = i; 
	current_panel->top_file = i;
	paint_dir (current_panel);
	panel_refresh (current_panel);
	select_item (current_panel);
    }
    return 1;
}

static void destroy_list()
{
    find_list *aux;

    while (list && list->up)
	list = list ->up;
    while (list){
	aux = list;
	list = list->down;
	free (aux);
    }
    list = NULL;
}

static void find_dialog ()
{
    char   *search_file = "*";
    char   *menu_text[] = {"   ChDir   "," Find Again",
			   "    Stop   ","    Quit   "};
    int    mimenu  = 1;
    int    running = 1;
    int    i, key, quit;

    scr_pos = 0;
    directories = (FStack *) malloc (sizeof (FStack));
    create_find_dialog();
    if (!do_load_find_menu (search_file))
	return;
    search_file = input_find (search_file);
    if (!search_file)
	return;
    wattrset (text_win, REVERSE_COLOR);
    wclr (text_win);
    init_find_menu (text_win, MAX_FIND_MENU, menu_text, REVERSE_COLOR );
    list = NULL;
    nodelay (stdscr, TRUE);
    run_find_menu (text_win, MAX_FIND_MENU, menu_text, mimenu,
		   INPUT_COLOR, REVERSE_COLOR );
    
    for (quit = 0; !quit;){
	switch (key = getch ()){
	case 'V':
	case KEY_PPAGE:
	    if (!running && list ) {
		list->selected = 0;
		for (i = 0 ; i<MAX_FIND_FLINES-1 && list->up; i++)
		    list = list->up;
		list->selected=1;
		repaint_list (REVERSE_COLOR, INPUT_COLOR);
	    }
	    break;

	case XCTRL('v'):
	case KEY_NPAGE:
	    if (!running && list ) {
		list->selected = 0;
		for (i = 0 ; i<MAX_FIND_FLINES-1 && list->down; i++)
		    list = list->down;
		list->selected=1;
		repaint_list (REVERSE_COLOR, INPUT_COLOR);
	    }
	    break;
    
	case XCTRL('p'):
	case KEY_UP: 
	    mimenu = 1;
	    if (!running && list && list->up){
		list->selected = 0;
		list = list ->up;
		list->selected = 1;
		repaint_list (REVERSE_COLOR, INPUT_COLOR);
	    }
	    break;

	case XCTRL('n'):
	case KEY_DOWN: 
	    mimenu = 1;
	    if (!running && list && list->down) {
		list->selected = 0;
		list = list->down;
		list->selected = 1;
		repaint_list (REVERSE_COLOR, INPUT_COLOR);
	    }
	    break;

	case XCTRL('b'):
	case KEY_LEFT:
	    if (mimenu > 1)
		mimenu--;
	    else
		mimenu = MAX_FIND_MENU;
	    break;

	case XCTRL('f'):
	case KEY_RIGHT: 
	    if (mimenu < MAX_FIND_MENU)
		mimenu++;
	    else
		mimenu = 1;
	    break;

	case 3:			/* C-c */
	case '\e':
	    quit = 1;
	    break;
	    
	case '\n':
	    switch (mimenu){
	    case 1:
		if (select_and_change_panel())
		    return ;
		break;
		
	    case 2:
		if (!do_load_find_menu (search_file))
		    return ;
		nodelay (stdscr, FALSE);
		search_file = input_find (search_file);
		if (search_file) {
		    running = 1;
		    if (0)
			destroy_list(); 
		    /* Borrar la lista y la pantalla */
		    /* en caso de no tener el acumula lista */
		    nodelay (stdscr, TRUE);
		}
		init_find_menu (text_win, MAX_FIND_MENU, menu_text,
				REVERSE_COLOR );
		break;
		
	    case 3:
		running = 0;
		for (i = 0, top_scr = list ; i < MAX_FIND_FLINES-1
		     && top_scr->up; i++)
		    top_scr = top_scr->up;
		list->selected = 1;
		repaint_list (REVERSE_COLOR, INPUT_COLOR);
		break;
		
	    case 4:
		quit = 1;
		break;
		
	    default: /* File Selection */
		if (select_and_change_panel())
		    return ;
		break;
	    }
	    break;
	    
	case ERR:
	default:
	    if (running){
		running = add_list (search_file);
		if (!running){
		    for (i = 0, top_scr = list ; i<MAX_FIND_FLINES-1
			 && top_scr->up; i++)
			top_scr = top_scr->up;
		    list->selected = 1;
		    repaint_list (REVERSE_COLOR, INPUT_COLOR);
		    nodelay (stdscr ,FALSE);
		}
	    }
	    continue;
	    break;
	}
	run_find_menu (text_win, MAX_FIND_MENU, menu_text, mimenu,
		       INPUT_COLOR, REVERSE_COLOR );
    }
    nodelay (stdscr, FALSE);
    destroy_list ();
    destroy_dialog ();
}

void do_find ()
{
    find_dialog ();
    refresh_screen ();
    nodelay (stdscr, FALSE);
}


