 /* Main program for the MouseLess Commander
   Copyright (C) 1994 Miguel de Icaza
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <ncurses.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#ifdef __svr4__
#include <dirent.h>
#else
#include <sys/dir.h>
#endif
#include <errno.h>
#include <pwd.h>
#include <unistd.h>
#include <fcntl.h>		/* For O_RDWR */
#include <signal.h>

#include "dir.h"
#include "color.h"
#include "panel.h"
#include "input.h"
#include "global.h"
#include "dialog.h"
#include "menu.h"
#include "file.h"
#include "main.h"
#include "util.h"
#include "win.h"
#include "user.h"
#include "mouse.h"

static char rcsid [] = "$Header: /usr/users/miguel/c/CVS/nc/main.c,v 1.15 1994/09/08 19:28:12 miguel Exp $";

/* The structures for the panels */
Panel left_panel, right_panel;

/* Pointers to the selected and unselected panel */
Panel *current_panel, *other_panel;

/* This holds the command line */
Input *cmdline;

/* Set when main loop should be terminated */
int   quit = 0;

/* If true, marking a files moves the cursor down */
int   mark_moves_down = 1;

/* If true, at startup the user-menu is invoked */
int   auto_menu = 0;

/* If true, after executing a command, wait for a keystroke */
int   pause_after_run = 1;

/* It true saves the setup when quitting */
int auto_save_setup = 0;

/* If true, next character is quoted */
int quote = 0;

/* If true use the internal viewer */
int use_internal_view = 1;

/* Have we shown the fast-reload warning in the past? */
int fast_reload_w = 0;

/* If true all typed chars go to the incremental search routine */
int searching = 0;

char *prompt = "$ ";

/* For slow terminals */
int force_slow = 0;

/* use mouse? */
int use_mouse_p = 1;

/* Controls screen clearing before an exec */
int clear_before_exec = 1;

/* Asks for confirmation before deleting a file */
int confirm_delete = 1;

/* Asks for confirmation before overwriting a file */
int confirm_overwrite = 1;

/* WINDOW handles for the command line and the function keys */
WINDOW *cmdline_win;
WINDOW *fkeys;
WINDOW *clean_screen;

/* The currently selected file */
file_entry *selection;

/* The home directory */
char *home_dir;

/* The value of the other directory, only used when loading the setup */
char *other_dir = 0;

/* If true, then print on stdout the last directory we were at */
int print_last_wd = 0;

/* Ugly hack in order to distinguish between left and right panel in menubar */
int is_right;
#define MENU_PANEL  (is_right ? &right_panel : &left_panel)
#define OTHER_PANEL (is_right ? &left_panel : &right_panel)

/* Quick search status */
char search_buffer [256] = { 0 };
char cmd_buf [512];

extern char *sys_errlist [];

void save_setup_cmd ();
void menu_cmd ();

inline void unselect_item (Panel *panel)
{
    set_attr (cpanel, 0, selection->f.marked);
    repaint_file (cpanel, cpanel->selected, RP_SETPOS);
    panel_refresh (cpanel);
}

#define SELECT_ITEM(panel) \
    selection = &current_panel->dir.list [current_panel->selected]; \
    set_attr (cpanel, 1, selection->f.marked);

/* This function sets the selection variable and redisplays the data */
inline void select_item (Panel *panel)
{
    SELECT_ITEM (panel);
    repaint_file (cpanel, cpanel->selected, RP_SETPOS);
    panel_refresh (cpanel);
    display_mini_info (cpanel);
}

#ifdef use_mouse
int panel_event    (Gpm_Event *event, Panel *panel);
int menu_bar_event (Gpm_Event *event, void *);
#endif
void init_panels ()
{
    init_panel (&left_panel, 0, 0, COLS/2, LINES-2);
    left_panel.active = 1;
    left_panel.count  = do_load_dir (&left_panel.dir, left_panel.sort_type,
				     left_panel.reverse);
    if (other_dir)
	chdir (other_dir);
    init_panel (&right_panel, COLS/2, 0, COLS, LINES-2);
    right_panel.count = do_load_dir (&right_panel.dir, right_panel.sort_type,
				     right_panel.reverse);
    if (other_dir)
	chdir (left_panel.cwd);

    current_panel = &left_panel;
    other_panel   = &right_panel;
    
    select_item (current_panel);/* Inits selection variable */
    display_mini_info (cpanel);	/* It needs selection defined */
    display_mini_info (opanel);	/* Same here */

    /* Now, update the screen */
    paint_panel (&left_panel);
    paint_panel (&right_panel);

    /* Panel events */
    push_event (2, 2, COLS/2-2, LINES-2, (mouse_h) panel_event,
		(void *) &left_panel);
    push_event (COLS/2+2, 2, COLS-1, LINES-2, (mouse_h) panel_event,
		(void *) &right_panel);

    /* Menu bar event */
    push_event (1, 1, COLS, 1, (mouse_h) menu_bar_event, 0);
}

void init_entry ()
{
    cmdline_win = newwin (1, COLS, LINES-2, 0);
    wprintw (cmdline_win, prompt);
    raw ();
    noecho ();
    keypad (stdscr, TRUE);
    wrefresh (cmdline_win);
}

void try_to_select (Panel *panel, char *name)
{
    Xtry_to_select (panel, name);
    SELECT_ITEM (panel);
    display_mini_info (panel);
}

/* This routine reloads the directory in both panels. It tries to
** select current_file in current_panel and other_file in other_panel.
** If current_file == -1 then it automatically sets current_file and
** other_file to the currently selected files in the panels.
*/
void update_panels (int force_update, char *current_file, char *other_file)
{
    int free_pointers = 0;

    if (force_update == UP_RELOAD){
	bzero (&(cpanel->dir_stat), sizeof (cpanel->dir_stat));
	bzero (&(opanel->dir_stat), sizeof (opanel->dir_stat));
    }
    
    /* If current_file == -1 (an invalid pointer, then preserve selection */
    if (current_file == UP_KEEPSEL){
	free_pointers = 1;
	current_file = strdup (cpanel->dir.list [cpanel->selected].fname);
	other_file   = strdup (opanel->dir.list [opanel->selected].fname);
    }
    panel_reload (cpanel);
    try_to_select (cpanel, current_file);
    panel_reload (opanel);
    try_to_select (opanel, other_file);
    paint_dir (cpanel);
    paint_dir (opanel);
    if (free_pointers){
	free (current_file);
	free (other_file);
    }
    chdir (cpanel->cwd);
}

/* Sets up the terminal before executing a program */
void pre_exec ()
{
    if (clear_before_exec){
	wstandend (clean_screen);
	werase (clean_screen);
	touchwin (clean_screen);
	wrefresh (clean_screen);
    } else {
	printf ("\n\n");
    }
    keypad (stdscr, FALSE);
    reset_shell_mode ();
    if (use_mouse_p)
	shut_mouse ();
    endwin ();
}

/* Restores the terminal after an exec, see execute for extra post-exec code */
void post_exec ()
{
    reset_prog_mode ();
    flushinp ();
    cbreak ();
    keypad (stdscr, TRUE);
    if (use_mouse_p)
	init_mouse ();
}

/* Save current stat of directories to avoid reloading the panels */
/* when no modifications have took place */
void save_cwds_stat ()
{
    if (fast_reload){
	stat (cpanel->cwd, &(cpanel->dir_stat));
	stat (opanel->cwd, &(opanel->dir_stat));
    }
}

void execute (char *command)
{
#ifdef OLD_SYSTEM_CODE
    char buf [4096];
#endif
    struct passwd *pw_data;

    pw_data = getpwuid (geteuid ());

#ifdef OLD_SYSTEM_CODE
    sprintf (buf, "%s -c \"%s\"", pw_data->pw_shell, command);
#endif
    
    save_cwds_stat ();
    pre_exec ();
#ifdef OLD_SYSTEM_CODE
    system (buf);
#else
    my_system (pw_data->pw_shell, command);
#endif
    if (pause_after_run){
	printf ("Press any key to continue...");
	fflush (stdout);
	cbreak ();
	mi_getch ();
    }
    post_exec ();
    update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
    refresh_screen ();
}

/* Changes the currently selected panel to the other panel */
void change_panel ()
{
    if (other_panel->view_type == view_info)
	return;

    if (other_panel->view_type & VIEW_DISABLED)
	return;
    
    current_panel->active = 0;
    show_dir (current_panel);
    unselect_item (current_panel);
    other_panel = current_panel;
    if (current_panel == &left_panel)
	current_panel = &right_panel;
    else
	current_panel = &left_panel;
    current_panel->active = 1;
    if (chdir (current_panel->cwd) != 0){
	message (1, " Error ", " Can't chdir to %s ", current_panel->cwd);
    }
    show_dir (current_panel);
    select_item (current_panel);
}

int quit_cmd ()
{
    if (query_dialog (" The MouseLess Commander  ",
		      " Do you really want to quit the Mouse-less commander? ",
		      0, 2, " Yes ", " No ") == 0)
	quit = 1;
    return quit;
}

/* Returns the number of items in the given panel */
int ITEMS (Panel *p)
{
    if (p->view_type == view_brief)
	return p->lines * 2;
    else
	return p->lines;
}

void move_down ()
{
    if (cpanel->selected+1 == cpanel->count)
	return;
    
    unselect_item (cpanel);
    cpanel->selected++;
    
    if (cpanel->selected - cpanel->top_file == ITEMS (cpanel)){
	/* Scroll window half screen */
	cpanel->top_file += ITEMS (cpanel)/2;
	paint_dir (cpanel);
	select_item (cpanel);
    } 
    select_item (cpanel);
    panel_refresh (cpanel);
}

void move_up ()
{
    if (cpanel->selected == 0)
	return;
    
    unselect_item (cpanel);
    cpanel->selected--;
    if (cpanel->selected < cpanel->top_file){
	/* Scroll window half screen */
	cpanel->top_file -= ITEMS (cpanel)/2;
	if (cpanel->top_file < 0) cpanel->top_file = 0;
	paint_dir (cpanel);
    } 
    select_item (cpanel);
    panel_refresh (cpanel);
}

void move_home ()
{
    if (cpanel->selected == 0)
	return;
    unselect_item (cpanel);
    cpanel->top_file = 0;
    cpanel->selected = 0;
    paint_dir (cpanel);
    select_item (cpanel);
    panel_refresh (cpanel);
}

void move_end ()
{
    if (cpanel->selected == cpanel->count-1)
	return;
    unselect_item (cpanel);
    cpanel->selected = cpanel->count-1;
    cpanel->top_file = cpanel->selected - ITEMS (cpanel) + 1;
    if (cpanel->top_file < 0) cpanel->top_file = 0;
    paint_dir (cpanel);
    select_item (cpanel);
    panel_refresh (cpanel);
}

void prev_page ()
{
    unselect_item (cpanel);
    cpanel->selected -= ITEMS (cpanel);
    cpanel->top_file -= ITEMS (cpanel);
    if (cpanel->selected < 0) cpanel->selected = 0;
    if (cpanel->top_file < 0) cpanel->top_file = 0;
    paint_dir (cpanel);
    select_item (cpanel);
    panel_refresh (cpanel);
}

void next_page ()
{
    unselect_item (cpanel);
    cpanel->selected += ITEMS (cpanel);
    cpanel->top_file += ITEMS (cpanel);
    if (cpanel->selected >= cpanel->count)
	cpanel->selected = cpanel->count - 1;
    if (cpanel->top_file >= cpanel->count)
	cpanel->top_file = cpanel->count - 1;
    paint_dir (cpanel);
    select_item (cpanel);
    panel_refresh (cpanel);
}

void do_mark_file (int do_move)
{
    if (!S_ISDIR (selection->buf.st_mode)){
	selection->f.marked = selection->f.marked ? 0 : 1;
	set_attr  (cpanel, 1, selection->f.marked);
	repaint_file (cpanel, cpanel->selected, RP_SETPOS);
	if (selection->f.marked){
	    cpanel->marked++;
	    cpanel->total += selection->buf.st_size;
	    set_colors (cpanel);
	    if (cpanel->marked == 1)
		mvwprintw (cpanel->win_file, cpanel->lines+3, 1,
			   "%-*s", cpanel->cols, "", COLS/2-2);
	} else {
	    cpanel->marked--;
	    cpanel->total -= selection->buf.st_size;
	}
    }
    if (mark_moves_down && do_move)
	move_down ();
    display_mini_info (cpanel);
    panel_refresh (cpanel);
}

void mark_file ()
{
    do_mark_file (1);
}

/* This routine untouches the first line on both panels in order */
/* to avoid the refreshing the menu bar */
void untouch_bar ()
{
    touchwin (cpanel->win_file);
    touchwin (opanel->win_file);
    wtouchln (cpanel->win_file, 0, 1, 0);
    wtouchln (opanel->win_file, 0, 1, 0);
    panel_refresh (cpanel);
    panel_refresh (opanel);
}

void clr_scr ()
{
    wstandend (clean_screen);
    wclear (clean_screen);
    touchwin (clean_screen);
    wrefresh (clean_screen);
}

void repaint_screen (int clear)
{
    if (clear){
	clr_scr ();
    }
    touchwin (current_panel->win_file);
    panel_refresh (current_panel);
    touchwin (other_panel->win_file);
    panel_refresh (other_panel);
    
    touchwin (cmdline_win);
    touchwin (fkeys);
    
    wrefresh (cmdline_win);
    wrefresh (fkeys);
}

void repaint_screen_cmd ()
{
    repaint_screen (RP_CLEAR);
}

void only_refresh_screen ()
{
    panel_refresh (current_panel);
    panel_refresh (other_panel);
    wrefresh (cmdline_win);
    wrefresh (fkeys);
}

void refresh_screen ()
{
    touchwin (current_panel->win_file);
    touchwin (other_panel->win_file);
    touchwin (cmdline_win);
    touchwin (fkeys);
    only_refresh_screen ();
}

void help_cmd ()
{
    interactive_display (HELPFILE);
}

int do_cd (char *new_dir)
{
    int  ret;
    DIR  *dirp;
    char *directory;
    char temp [MAXPATHLEN];
	
    if (!strcmp (new_dir, "-")){
	strcpy (temp, cpanel->lwd);
	new_dir = temp;
	strcpy (cpanel->lwd, cpanel->cwd);
    } else {
	strcpy (cpanel->lwd, cpanel->cwd);
    }
    
    directory = *new_dir ? new_dir : home_dir;
    dirp = opendir (directory);
    if (!dirp)
	return 0;
    closedir (dirp);

    ret = chdir (directory);
    if (ret == -1)
	return 0;
    
    clean_dir (&cpanel->dir, cpanel->count);
    cpanel->count = do_load_dir (&cpanel->dir, cpanel->sort_type,
				 cpanel->reverse);
    cpanel->top_file = 0;
    cpanel->selected = 0;
    cpanel->marked = 0;
    cpanel->total = 0;
    getwd (cpanel->cwd);
    return 1;
}

void action ()
{
    char *old_dir;

    if (S_ISDIR (selection->buf.st_mode) || link_isdir (selection)){
	if (!strcmp (selection->fname, ".."))
	    old_dir = strdup (cpanel->cwd);
	else
	    old_dir = 0;
	if (!do_cd (selection->fname))
	    return;
	try_to_select (cpanel, old_dir);
	free (old_dir);
	paint_panel (cpanel);
	select_item (cpanel);
	return;
    }
    if (is_exe (selection->buf.st_mode) && if_link_is_exe (selection)){
	/* Need to append "./" to the command */
	execute (selection->fname);
	return;
    }
    regex_command (selection->fname);
}

void enter ()
{
    if (strlen (cmdline->buffer)){
	if (strncmp (cmdline->buffer, "cd ", 3) == 0 ||
	    strcmp (cmdline->buffer, "cd") == 0){
	    if (cmdline->buffer [2] == 0)
		cmdline->buffer [3] = 0;
	    if (!do_cd (&cmdline->buffer [3])){
		message (1, " Error ", " Can't chdir to '%s' ",
			 &cmdline->buffer [3]);
		return;
	    }
	    paint_panel (cpanel);
	    select_item (cpanel);
	    new_input (cmdline);
	} else {
	    execute (cmdline->buffer);
	    new_input (cmdline);
	}
	return;
    }
    action ();
}

void do_search (int c_code)
{
    int l, i;

    l = strlen (search_buffer);
    if (l && (c_code == 8 || c_code == 0177 || c_code == KEY_BACKSPACE))
	search_buffer [--l] = 0;
    else {
	search_buffer [l] = c_code & 0x7f;
	search_buffer [l+1] = 0;
	l++;
    }
    for (i = 0; i < cpanel->count; i++){
	if (strncmp (cpanel->dir.list [i].fname, search_buffer, l) == 0){
	    cpanel->selected = i;
	    cpanel->top_file = cpanel->selected - cpanel->lines/2;
	    if (cpanel->top_file < 0)
		cpanel->top_file = 0;

	    select_item (cpanel);
	    paint_panel (cpanel);
	    break;
	}
    }
}

void start_search ()
{
    searching = 1;
    search_buffer [0] = 0;
}

#ifdef use_mouse

static int mouse_marking = 0;

void mouse_toggle_mark ()
{
    do_mark_file (0);
    mouse_marking = selection->f.marked;
}

void mouse_set_mark ()
{
    if (mouse_marking && !selection->f.marked)
	do_mark_file (0);
    else if (!mouse_marking && selection->f.marked)
	do_mark_file (0);
}

inline int mark_if_marking (Gpm_Event *event)
{
    if (event->buttons & GPM_B_RIGHT){
	if (event->type & GPM_DOWN)
	    mouse_toggle_mark ();
	else
	    mouse_set_mark ();
	return 1;
    }
    return 0;
}
    
int panel_event (Gpm_Event *event, Panel *panel)
{
    int index;
    
    if ((event->type & (GPM_DOWN|GPM_DRAG))){

	if (panel != current_panel)
	    change_panel ();

	if (event->y <= 0){
	    mark_if_marking (event);
	    move_up ();
	    return MOU_REPEAT;
	}

	if (!((panel->top_file + event->y <= panel->count) &&
	      event->y <= panel->lines)){
	    mark_if_marking (event);
	    move_down ();
	    return MOU_REPEAT;
	} else
	    index = panel->top_file + event->y - 1;

	if (index != panel->selected){
	    unselect_item (panel);
	    panel->selected = index;
	    select_item (panel);
	}
	if (!mark_if_marking (event))
	    panel_refresh (panel);
    } else if ((event->type & (GPM_UP|GPM_DOUBLE)) == (GPM_UP|GPM_DOUBLE))
	action ();
    return MOU_NORMAL;
}

int menu_bar_event (Gpm_Event *event, void *x)
{
    if (event->type != GPM_DOWN)
	return MOU_NORMAL;
    menu_cmd ();
    return MOU_NORMAL;
}
#endif

/* If the key pressed is not bound to any command, this function is called */
void default_key (int c_code)
{
    if (searching)
	do_search (c_code);
    else {
	handle_char (cmdline, c_code);
    }
}

void view_cmd ()
{
    static char *viewer = 0;

    if (use_internal_view){
	view (selection->fname);
	refresh_fn = refresh_screen;
	repaint_screen (RP_NOCLEAR);
    } else {
	if (!viewer){
	    viewer = getenv ("PAGER");
	    if (!viewer)
		viewer = "view";
	}
	sprintf (cmd_buf, "%s %s", viewer, selection->fname);
	execute (cmd_buf);
    }
}

void do_edit (const char *what)
{
    static char *editor = 0;

    if (!editor){
	editor = getenv ("EDITOR");
	if (!editor)
	    editor = "vi";
    }
    sprintf (cmd_buf, "%s %s", editor, what);
    execute (cmd_buf);
}

void edit_cmd ()
{
    do_edit (selection->fname);
}

/* Returns currently selected file or the first marked file if there is one */
char *get_file (Panel *panel)
{
    int i;
    
    if (panel->marked){
	for (i = 0; i < panel->count; i++)
	    if (panel->dir.list [i].f.marked)
		return panel->dir.list [i].fname;
    } else
	return panel->dir.list [panel->selected].fname;
    fprintf (stderr, " Internal error: get_file \n");
    mi_getch ();
    return "";
}

/* Clears all files in the panel, used only when one file was marked */

void unmark_file (Panel *panel)
{
    int i;

    if (!panel->marked)
	return;
    for (i = 0; i < panel->count; i++)
	panel->dir.list [i].f.marked = 0;
    panel->marked = 0;
    panel->total = 0;
}

int ok_to_copy (char *filename, char *dest, int dest_is_file)
{
    char *fullname;

    fullname = dest_is_file ? dest : get_full_name (dest, filename);
    
    if (confirm_overwrite && !access (fullname, 0)){
	if (query_dialog (" Copy ",
			  " Destination file already exists, overwrite? ",
			  0, 2, " Yes ", " No ") == 1){
	    if (dest_is_file) free (fullname);
	    return 0;
	} 
    }
    if (!dest_is_file) free (fullname);
    return 1;
}

int ok_to_copy_all (char *dir, char *file, int *skip_all)
{
    char *fullname = get_full_name (dir, file);
    int  value;
    char buf [512];
    
    *skip_all = 0;

    sprintf (buf, "\n File %s exists, overwrite? ", file);
    
    if (confirm_overwrite && !access (fullname, 0)){
	free (fullname);
	value = query_dialog (" Copy ", buf, 0, 3, " Yes ", " No ", " All ");
	if (value == 1)
	    return 0;

	if (value == 2)
	    *skip_all = 1;
    } else 
	free (fullname);
    return 1;
}

void copy_cmd ()
{
    char *dest;
    char *filename;
    int  i, c, stat_r;
    int  skip_all;
    struct stat s;

    if (cpanel->marked > 0)
	sprintf (cmd_buf, " Copy %d files to:", cpanel->marked);
    else {
	if (!S_ISREG (selection->buf.st_mode))
	    return;
	sprintf (cmd_buf," Copy \"%s\" file to:",
		 name_trunc(selection->fname, 30));
    }
    dest = input_dialog (" Copy ", cmd_buf, other_panel->cwd);
    if (!dest)
	return;
    
    stat_r = stat (dest, &s);
    save_cwds_stat ();
    if (S_ISREG (s.st_mode) || stat_r != 0){
	if (cpanel->marked > 1){
	    message (1, " Error ", "Can't copy multiple files to one file");
	    free (dest);
	    return;
	}
	filename = get_file (cpanel);
	if (ok_to_copy (filename, dest, 1))
	    copy_file_file (filename, dest);
	unmark_file (cpanel);
    } else {
	if (!S_ISDIR (s.st_mode)){
	    message (1, " Copy ", "Unknown copy destination");
	    free (dest);
	    return;
	}
	/* Destination is a directory */
	if (!cpanel->marked){
	    if (ok_to_copy (selection->fname, dest, 0))
		copy_file_dir (selection->fname, dest);
	} else {
	    nodelay (stdscr, TRUE);
	    skip_all = 0;
	    for (i = 0; i < cpanel->count; i++){
		if (cpanel->dir.list [i].f.marked){
		    if (!skip_all){
			int v;

			nodelay (stdscr, FALSE);
			v = ok_to_copy_all
			    (dest, cpanel->dir.list [i].fname, &skip_all);
			nodelay (stdscr, TRUE);
			if (!v)
			    continue;
		    }
		    copy_file_dir (cpanel->dir.list [i].fname, dest);
		    cpanel->dir.list [i].f.marked = 0;
		    cpanel->marked--;
		    cpanel->total -= cpanel->dir.list [i].buf.st_size;
		}
		if ((c = getch ()) != ERR)
		    if (c == '\e' || c == 3)
			break;
	    }
	    nodelay (stdscr, FALSE);
	}
    }
    free (dest);
    update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
    repaint_screen (RP_NOCLEAR);
}

void ren_cmd ()
{
    struct stat s;
    int i, c, stat_r;
    
    char *new_name = input_dialog (" Rename/Move ", "Enter rename mask:",
				   opanel->cwd);
    if (!new_name)
	return;
    if (strchr (new_name, '*') || strchr (new_name, '?')){
	/* Mask rename */
	message (0, " Missing feature ",
		 " Mask renaming not yet implemented, sorry ");
	free (new_name);
	return;
    }
    save_cwds_stat ();
    stat_r = stat (new_name, &s);
    if (S_ISREG (s.st_mode) || stat_r != 0){
	if (cpanel->marked <= 1){
	    move_file (get_file (cpanel), new_name);
	    unmark_file (cpanel);
	} else {
	    message (1," Error "," Can't rename multiple files to same name ");
	    free (new_name);
	    return;
	}
    } else if (!S_ISDIR (s.st_mode)){
	message (1, " Error ", " Destination is not a directory or a file ");
	free (new_name);
	return;
    } else {
	/* destination is a directory */
	if (cpanel->marked){
	    nodelay (stdscr, TRUE);
	    for (i = 0; i < cpanel->count; i++){
		if (cpanel->dir.list [i].f.marked){
		    move_file_dir (cpanel->dir.list [i].fname, new_name);
		}
		if ((c = getch ()) != ERR)
		    if (c == '\e' || c == 3)
			break;
	    }
	    nodelay (stdscr, FALSE);
	} else
	    move_file_dir (selection->fname, new_name);
    }
    free (new_name);
    update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
    repaint_screen (RP_NOCLEAR);
}

void errno_report (char *s)
{
    message (1, " Error ", " %s: %s ", s, unix_error_string (errno));
}

void do_link (int symbolic_link)
{
    struct stat s;
    char *dest, *src;
    int  stat_r;

    if (!symbolic_link){
	stat_r = stat (selection->fname, &s);
	if (stat_r != 0){
	    message (1, " Error ", " Couldn't stat %s ", selection->fname);
	    return;
	}
	if (!S_ISREG (s.st_mode))
	    return;
    }
    
    if (!symbolic_link){
	dest = input_dialog (" Link ", " Link file to: ", "");
	if (!dest)
	    return;
	save_cwds_stat ();
	if (-1 == link (selection->fname, dest))
	    errno_report ("link");
    } else {
	src = input_dialog (" Symlink ", "Symlink file: ", "");
	if (!src)
	    return;
	dest = input_dialog (" Symlink ", " -> To file: ", selection->fname);
	if (src){
	    save_cwds_stat ();
	    if (-1 == symlink (dest, src))
		errno_report ("symlink");
	    free (src);
	}
    }
    free (dest);
    update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
    repaint_screen (RP_NOCLEAR);
}

void link_cmd ()
{
    do_link (0);
}

void symlink_cmd ()
{
    do_link (1);
}

void mkdir_cmd ()
{
    char *dir = input_dialog (" Mkdir ", "Enter directoy name:" , "");
    if (!dir)
	return;

    save_cwds_stat ();
    if (mkdir (dir, 0777) == 0){
	update_panels (UP_OPTIMIZE, 0, 0);
	unselect_item (cpanel);
	try_to_select (cpanel, dir);
	repaint_screen (RP_NOCLEAR);
	select_item (cpanel);
	free (dir);
	return;
    }
    free (dir);
    message (1, " Error ", "  %s  ", unix_error_string (errno));
}

void delete_cmd ()
{
    int  i, result;

    save_cwds_stat ();
    if (cpanel->marked > 0){
	sprintf (cmd_buf, " Do you really want to delete %d files? ",
		 cpanel->marked);
	if (query_dialog (" Delete ", cmd_buf, 3, 2, " Yes ", " No ") == 0){
	    for (i = 0; i < cpanel->count; i++){
		if (cpanel->dir.list [i].f.marked)
		    erase_file (cpanel->dir.list [i].fname);
	    }
	} else
	    return;
    } else {
	if (confirm_delete){
	    sprintf (cmd_buf, "  Delete %s?  ", selection->fname);
	    result = query_dialog (" Delete ", cmd_buf, 3, 2, " Yes ", " No ");
	    if (result != 0)
		return;
	}
	if (S_ISDIR (selection->buf.st_mode))
	    erase_dir (selection->fname);
	else
	    erase_file (selection->fname);
    }
    update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
    repaint_screen (RP_NOCLEAR);
}

void find_cmd ()
{
    do_find ();
}

void ext_cmd ()
{
    char buffer [400];
    
    sprintf (buffer, "%s/.mc.ext", home_dir);
    do_edit (buffer);
}

void menu_edit_cmd ()
{
    char buffer [400];
    int dir;
    
    if (getuid () == 0)
	dir = query_dialog ("Menu file edit",
			    " Which menu file will you edit? ", 0, 3,
			      " Local ", " User ", " System Wide ");
    else
	dir = query_dialog ("Menu file edit",
			   " Which menu file will you edit? ", 0, 2,
			   " Local ", " User ");
    switch (dir){
    case 0:
	strcpy (buffer, "/.mc.menu");
	break;

    case 1:
	sprintf (buffer, "%s/.mc.menu", home_dir);
	break;

    case 2:
	strcpy (buffer, MENU);
	break;

    default:
	return;
    }
    do_edit (buffer);
}

void select_cmd ()
{
    char *reg_exp;
    int i;
    int c;

    reg_exp = input_dialog (" Select ","", "*");
    if (!reg_exp)
	return;
    for (i = 0; i < cpanel->count; i++){
	if (S_ISDIR (cpanel->dir.list [i].buf.st_mode))
	    continue;
	c = regexp_match (reg_exp, cpanel->dir.list [i].fname);
	if (c == -1){
	    message (1, " Error ", "  Malformed regular expression  ");
	    free (reg_exp);
	    return;
	}
	if (c){
	    if (!cpanel->dir.list [i].f.marked){
		cpanel->marked++;
		cpanel->total += cpanel->dir.list [i].buf.st_size;
	    }
	    cpanel->dir.list [i].f.marked = 1;
	}
    }
    paint_panel (cpanel);
    panel_refresh (cpanel);
    panel_refresh (cpanel);
    free (reg_exp);
}

void unselect_cmd ()
{
    char *reg_exp;
    int i;
    int c;

    reg_exp = input_dialog (" Unselect ","", "*");
    if (!reg_exp)
	return;
    for (i = 0; i < cpanel->count; i++){
	if (S_ISDIR (cpanel->dir.list [i].buf.st_mode))
	    continue;
	c = regexp_match (reg_exp, cpanel->dir.list [i].fname);
	if (c == -1){
	    message (1, " Error ", "  Malformed regular expression  ");
	    free (reg_exp);
	    return;
	}
	if (c){
	    if (cpanel->dir.list [i].f.marked){
		cpanel->marked--;
		cpanel->total -= cpanel->dir.list [i].buf.st_size;
	    }
	    cpanel->dir.list [i].f.marked = 0;
	}
    }
    paint_panel (cpanel);
    panel_refresh (cpanel);
    panel_refresh (cpanel);
    free (reg_exp);
}

/* Switches to full display panel, well, not so full */
void full_cmd ()
{
    Panel *p = MENU_PANEL;

    if (p->view_type & VIEW_DISABLED)
	return;

    full_frame (p);
    p->view_type = view_full;
    left_panel.view_type &= (~VIEW_DISABLED);
    right_panel.view_type &= (~VIEW_DISABLED);
    paint_panel (p);
}

/* Switches the panel to brief directory listing */
void brief_cmd ()
{
    Panel *p = MENU_PANEL;

    if (p->view_type & VIEW_DISABLED)
	return;

    brief_frame (p);
    p->view_type = view_brief;
    left_panel.view_type &= (~VIEW_DISABLED);
    right_panel.view_type &= (~VIEW_DISABLED);
    paint_panel (p);
}

void info_cmd ()
{
    
}

/* Switches the panel to long display (like ls -l) */
void long_cmd ()
{
    Panel *p = MENU_PANEL;

    if (p->view_type & VIEW_DISABLED)
	return;
    
    long_frame (p); 
    p->view_type = view_long;
    if (p == &left_panel)
	right_panel.view_type |= VIEW_DISABLED;
    else
	left_panel.view_type  |= VIEW_DISABLED;
    paint_panel (p);
}

/* Panel sorting related routines */
void do_re_sort (Panel *panel)
{
    char *filename = strdup (selection->fname);
    int  i;
    
    unselect_item (panel);
    do_sort (&panel->dir, panel->sort_type, panel->count-1, panel->reverse);
    panel->selected = -1;
    for (i = panel->count; i; i--){
	if (!strcmp (panel->dir.list [i-1].fname, filename)){
	    panel->selected = i-1;
	    break;
	}
    }
    select_item (panel);
    paint_dir (panel);
    panel_refresh (panel);
}

void by_name_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->sort_type = (sortfn *) sort_name;
    do_re_sort (panel);
}

void by_ext_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->sort_type = (sortfn *) sort_ext;
    do_re_sort (panel);
}

void by_time_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->sort_type = (sortfn *) sort_time;
    do_re_sort (panel);
}

void by_size_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->sort_type = (sortfn *) sort_size;
    do_re_sort (panel);
}

void unsorted_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->sort_type = (sortfn *) unsorted;
    do_re_sort (panel);
}

void reverse_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->reverse = !panel->reverse;
    do_re_sort (panel);
}

void reread_cmd ()
{
    update_panels (UP_RELOAD, UP_KEEPSEL, 0);
    repaint_screen (RP_NOCLEAR);
}

menu_entry PanelMenu [] = {
    "  Full",  'F', full_cmd,
    "  Brief", 'B', brief_cmd,
    "  Long",  'L', long_cmd, 
/*    "  Info",  'I', info_cmd, */
    "", ' ', 0,
    "  Name",      'N', by_name_cmd,
    "  Extension", 'E', by_ext_cmd,
    "  Time",      'T', by_time_cmd,
    "  Size",      'S', by_size_cmd,
    "  Unsorted",  'U', unsorted_cmd,
    "  reVerse ",  'V', reverse_cmd,
    "", ' ', 0,
    "  Reread    C-r",XCTRL('R'), reread_cmd,
/*    "  Filter...", 'l', reread_cmd, */
};

menu_entry RightMenu [] = {
    "  Full",  'F', full_cmd,
    "  Brief", 'B', brief_cmd,
    "  Long",  'L', long_cmd, 
    "", ' ', 0,
    "  Name",      'N', by_name_cmd,
    "  Extension", 'E', by_ext_cmd,
    "  Time",      'T', by_time_cmd,
    "  Size",      'S', by_size_cmd,
    "  Unsorted",  'U', unsorted_cmd,
    "  reVerse ",  'V', reverse_cmd,
    "", ' ', 0,
    "  Reread    C-r",    XCTRL('R'), reread_cmd,
    /*
    "  Filter...", 'l', reread_cmd,
    */
};

menu_entry FileMenu [] = {
    "Help            F1", KEY_F(1), help_cmd,
    "User menu       F2", KEY_F(2), user_menu_cmd,
    "View            F3", KEY_F(3), view_cmd,
    "Edit            F4", KEY_F(4), edit_cmd,
    "Copy            F5", KEY_F(5), copy_cmd,
    "Link              ", 'L',      link_cmd,
    "SymLink           ", 'S',      symlink_cmd,
    "Rename/Move     F6", KEY_F(6), ren_cmd,
    "Mkdir           F7", KEY_F(7), mkdir_cmd,
    "Delete          F8", KEY_F(8), delete_cmd,
    "", ' ', 0,
    "Select group     +", '+', select_cmd,
    "Unselect group   \\", '\\', unselect_cmd,
    "", ' ', 0,
    "Quit           F10", KEY_F(10), (callfn) quit_cmd
};

menu_entry CmdMenu [] = {
    "Find file (test)", 'F', find_cmd,
    "Save setup",       'S', save_setup_cmd,
    "Extension file edit", 'E', ext_cmd,
    "Menu file edit",   'M', menu_edit_cmd,
};

/* Must keep in sync with the constants in menu_cmd */
menu_entry OptMenu [] = {
    "  show Backup files", 'b', toggle_show_backup,
    "  show Hidden files", 'H', toggle_show_hidden,
    "  Verbose operation", 'V', toggle_verbose,
    "  mark moves Down",   'D', toggle_mark_move_down,
    "  Pause after run",   'P', toggle_pause_after_run,
    "  show mini-Status",  'S', toggle_show_mini_status,
    "  shEll patterns",    'E', toggle_easy_patterns,
    "  Auto save setup",   'A', toggle_auto_save,
    "  auto Menus",        'M', toggle_auto_menu,
    "  use Internal view", 'I', toggle_internal,
    "  miX all files",     'X', toggle_mix_all_files,
    "  Fast dir-reload",   'F', toggle_fast_reload,
    "  Confirm delete",    'C', toggle_confirm_delete,
/*    "  Align extensions",  'A', toggle_align_extensions, */
};

#define menu_entries(x) sizeof(x)/sizeof(menu_entry)

Menu *MenuBar [5];

void init_menu ()
{
    MenuBar [0] = create_menu (" Left ", PanelMenu, menu_entries (PanelMenu));
    MenuBar [1] = create_menu (" File ", FileMenu, menu_entries (FileMenu));
    MenuBar [2] = create_menu (" Command ", CmdMenu, menu_entries (CmdMenu));
    MenuBar [3] = create_menu (" Options ", OptMenu, menu_entries (OptMenu));
    MenuBar [4] = create_menu (" Right ", RightMenu, menu_entries (PanelMenu));
}

void check_options_menu (int index, int flag)
{
    *OptMenu [index].text = flag ? '*' : ' ';
}

void check_menu_panel (Panel *panel, menu_entry PanelMenu [])
{
    PanelMenu [0].text [0] = panel->view_type == view_full ? '*' : ' ';
    PanelMenu [1].text [0] = panel->view_type == view_brief ? '*' : ' ';
    PanelMenu [2].text [0] = panel->view_type == view_long ? '*' : ' ';
/*    PanelMenu [3].text [0] = panel->view_type == view_info ? '*' : ' '; */

    PanelMenu[4].text[0]= panel->sort_type == (sortfn *)sort_name ? '*' : ' ';
    PanelMenu[5].text[0]= panel->sort_type == (sortfn *) sort_ext ? '*' : ' ';
    PanelMenu[6].text[0]= panel->sort_type == (sortfn *) sort_time ? '*' : ' ';
    PanelMenu[7].text[0]= panel->sort_type == (sortfn *) sort_size ? '*' : ' ';
    PanelMenu[8].text[0]= panel->sort_type == (sortfn *) unsorted ? '*' : ' ';
    PanelMenu[9].text[0]= panel->reverse == 1 ? '*' : ' ';
}

void menu_cmd ()
{
    int panel;

    check_options_menu (0, show_backups);
    check_options_menu (1, show_dot_files);
    check_options_menu (2, verbose);
    check_options_menu (3, mark_moves_down);
    check_options_menu (4, pause_after_run);
    check_options_menu (5, show_mini_info);
    check_options_menu (6, easy_patterns);
    check_options_menu (7, auto_save_setup);
    check_options_menu (8, auto_menu);
    check_options_menu (9, use_internal_view);
    check_options_menu (10, mix_all_files);
    check_options_menu (11, fast_reload);
    check_options_menu (12, confirm_delete);
    
    check_menu_panel (&left_panel, PanelMenu);
    check_menu_panel (&right_panel, RightMenu);
    panel = current_panel == &right_panel ? 4 : 0;

    push_frame (0, 0, 0);
    run_bar (stdscr, 0, 0, 80, 12, 5, panel, MenuBar, BAR_FILL, A_BOLD,
	     SELECTED_COLOR);
    pop_frame ();
}

/* Flag toggling functions */

void toggle_confirm_delete ()
{
    confirm_delete = !confirm_delete;
}

void toggle_fast_reload ()
{
    fast_reload = !fast_reload;
    if (fast_reload_w == 0 && fast_reload){
	message (0, " Information ",
		 " Using the fast reload option may not reflect the exact \n"
		 " directory contents. In this cases you'll need to do a  \n"
		 " manual reload of the directory. See the man page for   \n"
		 " the details.                                           ");
	fast_reload_w = 1;
    }
}

void toggle_mix_all_files ()
{
    mix_all_files = !mix_all_files;
    update_panels (UP_RELOAD, UP_KEEPSEL, 0);
}

void toggle_internal ()
{
    use_internal_view = !use_internal_view;
}

void toggle_auto_menu ()
{
    auto_menu = !auto_menu;
}

void toggle_show_backup ()
{
    show_backups = !show_backups;
    update_panels (UP_RELOAD, UP_KEEPSEL, 0);
}

void toggle_show_hidden ()
{
    show_dot_files = !show_dot_files;
    update_panels (UP_RELOAD, UP_KEEPSEL, 0);
}

void toggle_verbose ()
{
    verbose = !verbose;
}

void toggle_auto_save ()
{
    auto_save_setup = !auto_save_setup;
}

void toggle_mark_move_down ()
{
    mark_moves_down = !mark_moves_down;
}

void toggle_pause_after_run ()
{
    pause_after_run = !pause_after_run;
}

void toggle_show_mini_status ()
{
    if (show_mini_info){
	left_panel.lines += 2;
	right_panel.lines +=2;
    } else {
	left_panel.lines -= 2;
	right_panel.lines -= 2;
    }
    show_mini_info = !show_mini_info;
    paint_panel (cpanel);
    paint_panel (opanel);
}

void toggle_easy_patterns ()
{
    easy_patterns = !easy_patterns;
}

void toggle_align_extensions ()
{
    align_extensions = !align_extensions;
}

void dump_marked ()
{
    int i;
    
    system ("clear");
    move (1, 1);
    for (i = 1; i < 8; i++){
	attron (COLOR_PAIR (i));
	printw ("Este es el color %d\n", i);
    }
    getch ();
    fprintf (stderr, "Hay %d archivos marcados\n", cpanel->marked);
    for (i = 0; i < cpanel->count; i++){
	if (cpanel->dir.list [i].f.marked)
	    fprintf (stderr, "%s\n", cpanel->dir.list [i].fname);
    }
    getch ();
    repaint_screen (RP_NOCLEAR);
}

/* Select a file name in a panel */

void goto_top_file ()
{
    unselect_item (cpanel);
    cpanel->selected = cpanel->top_file;
    select_item (cpanel);
    panel_refresh (cpanel);
}

void goto_middle_file ()
{
    unselect_item (cpanel);
    cpanel->selected = cpanel->top_file + (ITEMS(cpanel)/2);
    if (cpanel->selected >= cpanel->count)
	cpanel->selected = cpanel->count - 1;
    select_item (cpanel);
    panel_refresh (cpanel);
}

void goto_bottom_file ()
{
    unselect_item (cpanel);
    cpanel->selected = cpanel->top_file + ITEMS(cpanel)-1;
    if (cpanel->selected >= cpanel->count)
	cpanel->selected = cpanel->count - 1;
    select_item (cpanel);
    panel_refresh (cpanel);
}

/* Inserts the selected file name into the input line */
void copy_prog_name ()
{
    stuff (cmdline, selection->fname, 1);
}

/* Inserts the selected file name in the other panel into the input line */
void copy_other_prog_name ()
{
    char *k = opanel->dir.list [opanel->selected].fname;

    if ((opanel->view_type==view_info) || (opanel->view_type & VIEW_DISABLED))
	return;

    stuff (cmdline, k, 1);
}

void copy_current_pathname ()
{
    stuff (cmdline, cpanel->cwd, 0);
}

void copy_other_pathname ()
{
    stuff (cmdline, opanel->cwd, 0);
}

void copy_tagged (Panel *panel)
{
    int i;

    if (panel->marked){
	for (i = 0; i < cpanel->count; i++)
	    if (cpanel->dir.list [i].f.marked)
		stuff (cmdline, panel->dir.list [i].fname, 1);
    } else 
	stuff (cmdline, panel->dir.list [panel->selected].fname, 1);
}
    
void copy_current_tagged ()
{
    copy_tagged (cpanel);
}

void copy_other_tagged ()
{
    copy_tagged (opanel);
}

void suspend_cmd ()
{
    save_cwds_stat ();
    pre_exec ();
    kill (getpid (), SIGTSTP);
    post_exec ();
    update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
/*    repaint_screen (RP_CLEAR); */
}

void quote_next ()
{
    quote = 1;
}

key_map ctl_x_map [] = {
    XCTRL('c'),   (callfn) quit_cmd,
    'p',          copy_current_pathname,
    XCTRL('p'),   copy_other_pathname,
    't',          copy_current_tagged,
    XCTRL('t'),   copy_other_tagged,
    'l',          link_cmd,
    's',          symlink_cmd,
    'o',          change_panel,
    0, 0,
};

void ctl_x_cmd ()
{
    int i;
    int key = mi_getch ();

    for (i = 0; i < ctl_x_map [i].key_code; i++){
	if (key == ctl_x_map [i].key_code){
	    (*ctl_x_map [i].fn)(key);
	    break;
	}
    }
}

key_map default_map [] = {
    KEY_F(10), 	(callfn) quit_cmd,
    KEY_F(1),   help_cmd,
    KEY_F(2),   user_menu_cmd, 
    KEY_F(3),   view_cmd,
    KEY_F(4),   edit_cmd,
    KEY_F(5),   copy_cmd,
    KEY_F(6),   ren_cmd,
    KEY_F(7),   mkdir_cmd,
    KEY_F(8),   delete_cmd,
    KEY_F(9),   menu_cmd,
    KEY_F(12),  dump_marked,
    '+',        select_cmd,
    '\\',       unselect_cmd,
    '\t',  	change_panel,
    '\n',       enter,		/* Enter */
    KEY_DOWN,   move_down,
    KEY_UP, 	move_up,
    KEY_IC,     mark_file,
    KEY_HOME,	move_home,
    KEY_C1,     move_end,
    KEY_END,    move_end,
    KEY_A1,     move_home,
    KEY_NPAGE,  next_page,
    KEY_PPAGE,  prev_page,
    XCTRL('l'), repaint_screen_cmd, /* C-l */

    /* Copy usefull information to the command line */
    ALT('\n'),  copy_prog_name,
    ALT('\t'),  copy_other_prog_name,
    ALT('a'),   copy_current_pathname,
    ALT('A'),   copy_other_pathname,

    /* To quickly move in the panel */
    ALT('g'),   goto_top_file,
    ALT('h'),   goto_middle_file,
    ALT('j'),   goto_bottom_file,
    
    /* Emacs-like bindings */
    XCTRL('v'), next_page,      /* C-v like emacs */
    ALT('v'),   prev_page,	/* M-v like emacs */
    XCTRL('p'), move_up,	/* C-p like emacs */
    XCTRL('n'), move_down,	/* C-n like emacs */
    XCTRL('q'), quote_next,	/* C-q like emacs */
    XCTRL('s'), start_search,	/* C-s like emacs */
    ALT('s'),   start_search,	/* M-s not like emacs */
    
    /* Suspend */
    XCTRL('z'), suspend_cmd,

    XCTRL('t'), mark_file,

    /* Panel refresh */
    XCTRL('r'), reread_cmd,

    /* Control-X keybindings */
    XCTRL('x'), ctl_x_cmd,
    
    /* Default key handling */
    0,          default_key,
};

void init_labels ()
{
    fkeys   = new_fkey ();
    define_label (fkeys, 1, "Help", help_cmd);
    define_label (fkeys, 2, "Menu", user_menu_cmd);
    define_label (fkeys, 3, "View", view_cmd);
    define_label (fkeys, 4, "Edit", edit_cmd);
    define_label (fkeys, 5, "Copy", copy_cmd);
    define_label (fkeys, 6, "RenMov", ren_cmd);
    define_label (fkeys, 7, "Mkdir", mkdir_cmd);
    define_label (fkeys, 8, "Delete", delete_cmd);
    define_label (fkeys, 9, "PullDn", menu_cmd);
    define_label_quit (fkeys, 10, "Quit", quit_cmd);
    
    /* workaround ncurses 1.8.5 bug */
    wmove (fkeys, 0, 0);
    waddch (fkeys, '1');
    
    wrefresh (fkeys);
}

void save_setup_cmd ()
{
    save_setup ();
    message (0, " Setup ", " Setup saved to ~/.mc.ini ");
}

void do_nc ()
{
    int key;
    int i;

    refresh_fn = refresh_screen;
    if (COLS < 70 || LINES < 12){
	fprintf (stderr, "Screen too small: you need at leas 70x12\n");
	return;
    }

    /* Call all the inits */
    clean_screen = newwin (0, 0, 0, 0);
    home_dir = getenv ("HOME");
    home_dir = home_dir ? home_dir : "/";

    init_uid_gid_cache ();
    
    load_setup ();
    if (baudrate () <= 9600 || force_slow)
	verbose = 0;
    init_colors ();
    if (use_mouse_p)
	init_mouse ();
    init_labels ();
    init_key ();
    init_entry ();
    init_panels ();
    init_menu ();
    cmdline = create_input (strlen (prompt), 0, cmdline_win, 0,
			    COLS-strlen (prompt), "");

    if (auto_menu)
	user_menu_cmd ();

    /* Main program loop */
    for (quit = 0; !quit;){
	key = mi_getch ();
	if (quote){
	    default_key (key);
	    quote = 0;
	    continue;
	}
	for (i = 0; default_map [i].key_code; i++){
	    if (key == default_map [i].key_code){
		searching = 0;
		(*default_map [i].fn)(key);
		break;
	    }
	}
	if (default_map [i].key_code)
	    continue;

	/* Default key handling */
	(*default_map [i].fn)(key);
	wrefresh (cmdline_win);
    }
    destroy_input (cmdline, IN_NORMAL);
    if (auto_save_setup)
	save_setup ();
    clr_scr ();
    keypad (cmdline_win, FALSE);
    keypad (stdscr, FALSE);
    reset_shell_mode ();
    noraw ();
    if (use_mouse_p)
	shut_mouse ();
}

void version ()
{
    int mouse_support = 
#ifdef use_mouse
    1;
#else
    0;
#endif
    printf ("MouseLess Commander %s %s\n", VERSION,
	    mouse_support ? "with mouse support on Linux console" : "");
}

int main (int argc, char *argv [])
{
    extern int optind;
    char   *this_dir = 0;
    int    c;
    int    output1, output2;

    while ((c = getopt (argc, argv, "VbsdPcf")) != -1){
	switch (c){
	case 'V':
	    version ();
	    return 0;

	case 'b':
	    disable_colors = 1;
	    break;

	case 'c':
	    disable_colors = 0;
	    break;

	case 'f':
	    printf (
		    "Help file:            " HELPFILE "\n"
	            "Global defaults file: " PROFILE "\n"
		    "Global user menu:     " MENU "\n"
		    "Extensions file:      " EXT "\n");
	    exit (1);

	case 'P':
	    print_last_wd = 1;
	    break;
	    
	case 's':
	    force_slow = 1;	/* Currently it does nothing */
	    break;

	case 'd':
	    use_mouse_p = 0;
	    break;

	default:
	    version ();
	    printf ("Usage is:\n"
		    "mc [-bdPsV] [other_panel_dir]\n"
		    "-b Force black and white\n"
		    "-c Force color, only available if terminfo allows it\n"
		    "-d Disable mouse support\n"
		    "-f Print configured paths\n"
		    "-P At exit, print last working directory\n"
		    "-s Disables verbose operation (for slow terminals)\n"
		    "-V Print program version\n");
	    exit (1);
	}
    }

    /* sets the current dir and the other dir */
    for (; optind < argc; optind++){
	if (this_dir){
	    if (other_dir)
		break;
	    else {
		other_dir = argv [optind];
	    }
	} else {
	    this_dir = argv [optind];
	    getwd (right_panel.cwd);
	    other_dir = right_panel.cwd;
	}
    }
    if (this_dir)
	chdir (this_dir);
    if (print_last_wd){
	output1 = dup (1);
	close (1);
	output2 = open ("/dev/tty", O_RDWR);
    }
    save_stop_handler ();
    printf ("\ec");
    initscr ();
    do_nc ();
    endwin ();
    if (print_last_wd){
	close (output2);
	output2 = dup (output1);
	close (output1);
	printf ("%s", cpanel->cwd);
    }
    return 0;
}
