/* %Z%%M%	%I% %E% Copyright 1997 J. Schilling */
#ifndef lint
static	char __sccsid[] =
	"%Z%%M%	%I% %E% Copyright 1997 J. Schilling";
#endif
/*
 *	Interface for Linux generic SCSI implementation (sg).
 *
 *	This is the interface for the broken Linux SCSI generic driver.
 *	This is a hack, that tries to emulate the functionality
 *	of the scg driver.
 *
 *	Design flaws of the sg driver:
 *	-	cannot see if SCSI command could not be send
 *	-	cannot get SCSI status byte
 *	-	cannot get real dma count of tranfer
 *	-	cannot get number of bytes valid in auto sense data
 *	-	to few data in auto sense (CCS/SCSI-2/SCSI-3 needs >= 18)
 *
 *	This code contains support for the sg driver version 2 by
 *		H. Eifeld & J. Schilling
 *	Although this enhanced version has been announced to Linus and Alan,
 *	there was no reaction at all.
 *
 *	About half a year later there occured a version in the official
 *	Linux that was also called version 2. The interface of this version
 *	looks like a playground - the enhancements from this version are
 *	more or less useless for a portable real-world program.
 *
 *	With Linux 2.4 the official version of the sg driver is called 3.x
 *	and seems to be usable again. The main problem now is the curious
 *	interface that is provided to raise the DMA limit from 32 kB to a
 *	more reasonable value. To do this in a reliable way, a lot of actions
 *	are required.
 *
 *	Warning: you may change this source, but if you do that
 *	you need to change the _scg_version and _scg_auth* string below.
 *	You may not return "schily" for an SCG_AUTHOR request anymore.
 *	Choose your name instead of "schily" and make clear that the version
 *	string is related to a modified source.
 *
 *	Copyright (c) 1997 J. Schilling
 *
 *	Thanks to Alexander Kern <alex.kern@gmx.de> for the idea and first
 *	code fragments for supporting the CDROM_SEND_PACKET ioctl() from
 *	the cdrom.c kernel driver. Please note that this interface in prociple
 *	is completely unneeded but the Linux kernel is just a cluster of
 *	code and does not support planned orthogonal interface systems.
 *	For this reason we need CDROM_SEND_PACKET in order to work around a
 *	bug in the linux kernel that prevents to use PCATA drives because
 *	the kernel panics if you try to put ide-scsi on top of the PCATA
 *	driver.
 */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/version.h>

#ifndef LINUX_VERSION_CODE	/* Very old kernel? */
#	define LINUX_VERSION_CODE 0
#endif

#if LINUX_VERSION_CODE >= 0x01031a /* <linux/scsi.h> introduced in 1.3.26 */
#if LINUX_VERSION_CODE >= 0x020000 /* <scsi/scsi.h> introduced somewhere. */
/* Need to fine tune the ifdef so we get the transition point right. */
#include <scsi/scsi.h>
#else
#include <linux/scsi.h>
#endif
#else
#define __KERNEL__
#include <linux/fs.h>
#undef __KERNEL__
#include "block/blk.h"
#include "scsi/scsi.h"
#endif

#include "scsi/sg.h"

#undef sense			/* conflict in struct cdrom_generic_command */
#include <linux/cdrom.h>

#if	defined(CDROM_PACKET_SIZE) && defined(CDROM_SEND_PACKET)
#define	USE_ATA
#endif

/*
 *	Warning: you may change this source, but if you do that
 *	you need to change the _scg_version and _scg_auth* string below.
 *	You may not return "schily" for an SCG_AUTHOR request anymore.
 *	Choose your name instead of "schily" and make clear that the version
 *	string is related to a modified source.
 */
LOCAL	char	_scg_trans_version[] = "%M%-%I%";	/* The version for this transport*/
#ifdef	USE_ATA
LOCAL	char	_scg_atrans_version[] = "atapi-%I%";	/* The version for ATAPI	*/

LOCAL	int	scgo_aopen	__PR((SCSI *scgp, char *device));
LOCAL	int	scgo_aclose	__PR((SCSI *scgp));
LOCAL	long	scgo_amaxdma	__PR((SCSI *scgp, long amt));
LOCAL	BOOL	scgo_ahavebus	__PR((SCSI *scgp, int));
LOCAL	int	scgo_afileno	__PR((SCSI *scgp, int, int, int));
LOCAL	int	scgo_ainitiator_id __PR((SCSI *scgp));
LOCAL	int	scgo_aisatapi	__PR((SCSI *scgp));
LOCAL	int	scgo_areset	__PR((SCSI *scgp, int what));
LOCAL	int	scgo_asend	__PR((SCSI *scgp));

LOCAL scg_ops_t ata_ops = {
	scgo_asend,
	scgo_version,		/* Shared with SG driver */
	scgo_aopen,
	scgo_aclose,
	scgo_amaxdma,
	scgo_getbuf,		/* Shared with SG driver */
	scgo_freebuf,		/* Shared with SG driver */
	scgo_ahavebus,
	scgo_afileno,
	scgo_ainitiator_id,
	scgo_aisatapi,
	scgo_areset,
};
#endif	/* USE_ATA */

#ifndef	SCSI_IOCTL_GET_BUS_NUMBER
#define SCSI_IOCTL_GET_BUS_NUMBER 0x5386
#endif

/*
 * XXX There must be a better way than duplicating things from system include
 * XXX files. This is stolen from /usr/src/linux/drivers/scsi/scsi.h
 */
#ifndef	DID_OK
#define DID_OK          0x00 /* NO error                                */
#define DID_NO_CONNECT  0x01 /* Couldn't connect before timeout period  */
#define DID_BUS_BUSY    0x02 /* BUS stayed busy through time out period */
#define DID_TIME_OUT    0x03 /* TIMED OUT for other reason              */
#define DID_BAD_TARGET  0x04 /* BAD target.                             */
#define DID_ABORT       0x05 /* Told to abort for some other reason     */
#define DID_PARITY      0x06 /* Parity error                            */
#define DID_ERROR       0x07 /* Internal error                          */
#define DID_RESET       0x08 /* Reset by somebody.                      */
#define DID_BAD_INTR    0x09 /* Got an interrupt we weren't expecting.  */ 
#endif

/*
 *  These indicate the error that occurred, and what is available.
 */
#ifndef DRIVER_BUSY
#define DRIVER_BUSY         0x01
#define DRIVER_SOFT         0x02
#define DRIVER_MEDIA        0x03
#define DRIVER_ERROR        0x04
 
#define DRIVER_INVALID      0x05
#define DRIVER_TIMEOUT      0x06
#define DRIVER_HARD         0x07
#define DRIVER_SENSE        0x08
#endif

/*
 * XXX Should add extra space in buscookies and scgfiles for a "PP bus"
 * XXX and for two "ATAPI busses".
 */
#define	MAX_SCG		16	/* Max # of SCSI controllers */
#define	MAX_TGT		16
#define	MAX_LUN		8

#ifdef	USE_ATA
typedef struct {
	unsigned char   typ:4;
	unsigned char   host:8;
	unsigned char   bus:4;
}               __buscookies;
#endif

struct scg_local {
	int	scgfile;		/* Used for SG_GET_BUFSIZE ioctl()*/
	short	scgfiles[MAX_SCG][MAX_TGT][MAX_LUN];
	short	buscookies[MAX_SCG];
	int	pgbus;
	int	pack_id;		/* Should be a random number	*/
	int	drvers;
	int	isold;
	long	xbufsize;
	char	*xbuf;
	char	*SCSIbuf;

#ifdef	USE_ATA
/* # of virtual buses (schilly_host number) */
#define	MAX_SCHILLY_HOSTS	MAX_SCG
	__buscookies	bc[MAX_SCHILLY_HOSTS];
#endif
};
#define scglocal(p)	((struct scg_local *)((p)->local)) 

#ifdef	SG_BIG_BUFF
#define	MAX_DMA_LINUX	SG_BIG_BUFF	/* Defined in include/scsi/sg.h	*/
#else
#define	MAX_DMA_LINUX	(4*1024)	/* Old Linux versions		*/
#endif

#ifndef	SG_MAX_SENSE
#	define	SG_MAX_SENSE	16	/* Too small for CCS / SCSI-2	*/
#endif					/* But cannot be changed	*/

#if	!defined(__i386) && !defined(i386) && !defined(mc68000)
#define	MISALIGN
#endif
/*#define	MISALIGN*/
/*#undef	SG_GET_BUFSIZE*/

#if	defined(USE_PG) && !defined(USE_PG_ONLY)
#include "scsi-linux-pg.c"
#endif

#ifdef	MISALIGN
LOCAL	int	sg_getint	__PR((int *ip));
#endif
LOCAL	int	scgo_send	__PR((SCSI *scgp));
#ifdef	SG_IO
LOCAL	int	sg_rwsend	__PR((SCSI *scgp));
#endif
LOCAL	BOOL	sg_setup	__PR((SCSI *scgp, int f, int busno, int tgt, int tlun));
LOCAL	void	sg_initdev	__PR((SCSI *scgp, int f));
LOCAL	int	sg_mapbus	__PR((SCSI *scgp, int busno, int ino));
LOCAL	BOOL	sg_mapdev	__PR((SCSI *scgp, int f, int *busp, int *tgtp, int *lunp,
							int *chanp, int *inop));
#if defined(SG_SET_RESERVED_SIZE) && defined(SG_GET_RESERVED_SIZE)
LOCAL	long	sg_raisedma	__PR((SCSI *scgp, long newmax));
#endif
LOCAL	void	sg_settimeout	__PR((int f, int timeout));

/*
 * Return version information for the low level SCSI transport code.
 * This has been introduced to make it easier to trace down problems
 * in applications.
 */
LOCAL char *
scgo_version(scgp, what)
	SCSI	*scgp;
	int	what;
{
	if (scgp != (SCSI *)0) {
#ifdef	USE_PG
		/*
		 * If we only have a Parallel port or only opened a handle
		 * for PP, just return PP version.
		 */
		if (scglocal(scgp)->pgbus == 0 ||
		    (scg_scsibus(scgp) >= 0 &&
		     scg_scsibus(scgp) == scglocal(scgp)->pgbus))
			return (pg_version(scgp, what));
#endif
		switch (what) {

		case SCG_VERSION:
#ifdef	USE_ATA
			if (scgp->ops == &ata_ops)
				return (_scg_atrans_version);
#endif
			return (_scg_trans_version);
		/*
		 * If you changed this source, you are not allowed to
		 * return "schily" for the SCG_AUTHOR request.
		 */
		case SCG_AUTHOR:
			return (_scg_auth_schily);
		case SCG_SCCS_ID:
			return (__sccsid);
		case SCG_KVERSION:
			{
				static	char kv[16];
				int	n;

				if (scglocal(scgp)->drvers >= 0) {
					n = scglocal(scgp)->drvers;
					js_snprintf(kv, sizeof(kv),
                                        "%d.%d.%d",
                                        n/10000, (n%10000)/100, n%100);

					return (kv);
				}
			}
		}
	}
	return ((char *)0);
}

LOCAL int
scgo_open(scgp, device)
	SCSI	*scgp;
	char	*device;
{
		 int	busno	= scg_scsibus(scgp);
		 int	tgt	= scg_target(scgp);
		 int	tlun	= scg_lun(scgp);
	register int	f;
	register int	i;
	register int	b;
	register int	t;
	register int	l;
	register int	nopen = 0;
	char		devname[64];

	if (busno >= MAX_SCG || tgt >= MAX_TGT || tlun >= MAX_LUN) {
		errno = EINVAL;
		if (scgp->errstr)
			js_snprintf(scgp->errstr, SCSI_ERRSTR_SIZE,
				"Illegal value for busno, target or lun '%d,%d,%d'",
				busno, tgt, tlun);
		return (-1);
	}
	if (device != NULL && *device != '\0') {
#ifdef	USE_ATA
		if (strncmp(device, "ATAPI", 5) == 0) {
			scgp->ops = &ata_ops;
			return (SCGO_OPEN(scgp, device));
		}
#endif
	}

	if (scgp->local == NULL) {
		scgp->local = malloc(sizeof(struct scg_local));
		if (scgp->local == NULL)
			return (0);

		scglocal(scgp)->scgfile = -1;
		scglocal(scgp)->pgbus = -2;
		scglocal(scgp)->SCSIbuf = (char *)-1;
		scglocal(scgp)->pack_id = 5;
		scglocal(scgp)->drvers = -1;
		scglocal(scgp)->isold = -1;
		scglocal(scgp)->xbufsize = 0L;
		scglocal(scgp)->xbuf = NULL;

		for (b=0; b < MAX_SCG; b++) {
			scglocal(scgp)->buscookies[b] = (short)-1;
			for (t=0; t < MAX_TGT; t++) {
				for (l=0; l < MAX_LUN ; l++)
					scglocal(scgp)->scgfiles[b][t][l] = (short)-1;
			}
		}
	}

	if ((device != NULL && *device != '\0') || (busno == -2 && tgt == -2))
		goto openbydev;

	for (i=0; i < 32; i++) {
		js_snprintf(devname, sizeof(devname), "/dev/sg%d", i);
		f = open(devname, 2);
		if (f < 0) {
			/*
			 * Set up error string but let us clear it later
			 * if at least one open succeeded.
			 */
			if (scgp->errstr)
				js_snprintf(scgp->errstr, SCSI_ERRSTR_SIZE,
							"Cannot open '/dev/sg*'");
			if (errno != ENOENT && errno != ENXIO && errno != ENODEV) {
				if (scgp->errstr)
					js_snprintf(scgp->errstr, SCSI_ERRSTR_SIZE,
							"Cannot open '%s'", devname);
				return (0);
			}
		} else {
			if (sg_setup(scgp, f, busno, tgt, tlun))
				return (++nopen);
			if (busno < 0 && tgt < 0 && tlun < 0)
				nopen++;
		}
	}
	if (nopen > 0 && scgp->errstr)
		scgp->errstr[0] = '\0';
		
	if (nopen == 0) for (i=0; i <= 25; i++) {
		js_snprintf(devname, sizeof(devname), "/dev/sg%c", i+'a');
		f = open(devname, 2);
		if (f < 0) {
			/*
			 * Set up error string but let us clear it later
			 * if at least one open succeeded.
			 */
			if (scgp->errstr)
				js_snprintf(scgp->errstr, SCSI_ERRSTR_SIZE,
							"Cannot open '/dev/sg*'");
			if (errno != ENOENT && errno != ENXIO && errno != ENODEV) {
				if (scgp->errstr)
					js_snprintf(scgp->errstr, SCSI_ERRSTR_SIZE,
							"Cannot open '%s'", devname);
				return (0);
			}
		} else {
			if (sg_setup(scgp, f, busno, tgt, tlun))
				return (++nopen);
			if (busno < 0 && tgt < 0 && tlun < 0)
				nopen++;
		}
	}
	if (nopen > 0 && scgp->errstr)
		scgp->errstr[0] = '\0';

openbydev:
	if (device != NULL && *device != '\0') {
		f = open(device, 2);
/*		if (f < 0 && errno == ENOENT)*/
/*			goto openpg;*/

		if (f < 0) {
			/*
			 * The pg driver has the same rules to decide whether
			 * to use openbydev. If we cannot open the device, it
			 * makes no sense to try the /dev/pg* driver.
			 */
			if (scgp->errstr)
				js_snprintf(scgp->errstr, SCSI_ERRSTR_SIZE,
					"Cannot open '%s'",
					device);
			return (0);
		}

		if (!sg_mapdev(scgp, f, &busno, &tgt, &tlun, 0, 0)) {
			close(f);
			/*
			 * If sg_mapdev() failes, this may be /dev/pg* device.
			 */
			goto openpg;
		}

#ifdef	OOO
		if (scg_scsibus(scgp) < 0)
			scg_scsibus(scgp) = busno;
		if (scg_target(scgp) < 0)
			scg_target(scgp) = tgt;
		if (scg_lun(scgp) < 0)
			scg_lun(scgp) = tlun;
#endif

		scg_settarget(scgp, busno, tgt, tlun);
		if (sg_setup(scgp, f, busno, tgt, tlun))
			return (++nopen);
	}
openpg:
#ifdef	USE_PG
	nopen += pg_open(scgp, device);
#endif
	if (scgp->debug > 0) for (b=0; b < MAX_SCG; b++) {
		js_fprintf((FILE *)scgp->errfile,
			"Bus: %d cookie: %X\n",
			b, scglocal(scgp)->buscookies[b]);
		for (t=0; t < MAX_TGT; t++) {
			for (l=0; l < MAX_LUN ; l++) {
				if (scglocal(scgp)->scgfiles[b][t][l] != (short)-1) {
					js_fprintf((FILE *)scgp->errfile,
						"file (%d,%d,%d): %d\n",
						b, t, l, scglocal(scgp)->scgfiles[b][t][l]);
				}
			}
		}
	}
	return (nopen);
}

LOCAL int
scgo_close(scgp)
	SCSI	*scgp;
{
	register int	f;
	register int	b;
	register int	t;
	register int	l;

	if (scgp->local == NULL)
		return (-1);

	for (b=0; b < MAX_SCG; b++) {
		if (b == scglocal(scgp)->pgbus)
			continue;
		scglocal(scgp)->buscookies[b] = (short)-1;
		for (t=0; t < MAX_TGT; t++) {
			for (l=0; l < MAX_LUN ; l++) {
				f = scglocal(scgp)->scgfiles[b][t][l];
				if (f >= 0)
					close(f);
				scglocal(scgp)->scgfiles[b][t][l] = (short)-1;
			}
		}
	}
	if (scglocal(scgp)->xbuf != NULL) {
		free(scglocal(scgp)->xbuf);
		scglocal(scgp)->xbufsize = 0L;
		scglocal(scgp)->xbuf = NULL;
	}
#ifdef	USE_PG
	pg_close(scgp);
#endif
	return (0);
}

LOCAL BOOL
sg_setup(scgp, f, busno, tgt, tlun)
	SCSI	*scgp;
	int	f;
	int	busno;
	int	tgt;
	int	tlun;
{
	int	n;
	int	Chan;
	int	Ino;
	int	Bus;
	int	Target;
	int	Lun;
	BOOL	onetarget = FALSE;

#ifdef	SG_GET_VERSION_NUM
	if (scglocal(scgp)->drvers < 0) {
		scglocal(scgp)->drvers = 0;
		if (ioctl(f, SG_GET_VERSION_NUM, &n) >= 0) {
			scglocal(scgp)->drvers = n;
			if (scgp->overbose) {
				js_fprintf((FILE *)scgp->errfile,
					"Linux sg driver version: %d.%d.%d\n",
        				n/10000, (n%10000)/100, n%100);
			}
		}
	}
#endif
	if (scg_scsibus(scgp) >= 0 && scg_target(scgp) >= 0 && scg_lun(scgp) >= 0)
		onetarget = TRUE;

	sg_mapdev(scgp, f, &Bus, &Target, &Lun, &Chan, &Ino);

	/*
	 * For old kernels try to make the best guess.
	 */
	Ino |= Chan << 8;
	n = sg_mapbus(scgp, Bus, Ino);
	if (Bus == -1) {
		Bus = n;
		if (scgp->debug > 0) {
			js_fprintf((FILE *)scgp->errfile,
				"SCSI Bus: %d (mapped from %d)\n", Bus, Ino);
		}
	}

	if (Bus < 0 || Bus >= MAX_SCG || Target < 0 || Target >= MAX_TGT ||
						Lun < 0 || Lun >= MAX_LUN) {
		return (FALSE);
	}

	if (scglocal(scgp)->scgfiles[Bus][Target][Lun] == (short)-1)
		scglocal(scgp)->scgfiles[Bus][Target][Lun] = (short)f;

	if (onetarget) {
		if (Bus == busno && Target == tgt && Lun == tlun) {
			sg_initdev(scgp, f);
			scglocal(scgp)->scgfile = f;	/* remember file for ioctl's */
			return (TRUE);
		} else {
			scglocal(scgp)->scgfiles[Bus][Target][Lun] = (short)-1;
			close(f);
		}
	} else {
		sg_initdev(scgp, f);
		if (scglocal(scgp)->scgfile < 0)
			scglocal(scgp)->scgfile = f;	/* remember file for ioctl's */
	}
	return (FALSE);
}

LOCAL void
sg_initdev(scgp, f)
	SCSI	*scgp;
	int	f;
{
	struct sg_rep {
		struct sg_header	hd;
		unsigned char		rbuf[100];
	} sg_rep;
	int	n;

	/* Eat any unwanted garbage from prior use of this device */

	n = fcntl(f, F_GETFL);	/* Be very proper about this */
	fcntl(f, F_SETFL, n|O_NONBLOCK);

	fillbytes((caddr_t)&sg_rep, sizeof(struct sg_header), '\0');
	sg_rep.hd.reply_len = sizeof(struct sg_header);

	while (read(f, &sg_rep, sizeof(sg_rep)) >= 0 || errno != EAGAIN)
		;

	fcntl(f, F_SETFL, n);

	sg_settimeout(f, scgp->deftimeout);
}

LOCAL int
sg_mapbus(scgp, busno, ino)
	SCSI	*scgp;
	int	busno;
	int	ino;
{
	register int	i;

	if (busno >= 0 && busno < MAX_SCG) {
		/*
		 * SCSI_IOCTL_GET_BUS_NUMBER worked.
		 * Now we have the problem that Linux does not properly number
		 * SCSI busses. The Bus number that Linux creates really is
		 * the controller (card) number. I case of multi SCSI bus
		 * cards we are lost.
		 */
		if (scglocal(scgp)->buscookies[busno] == (short)-1) {
			scglocal(scgp)->buscookies[busno] = ino;
			return (busno);
		}
		if (scglocal(scgp)->buscookies[busno] != (short)ino)
			errmsgno(EX_BAD, "Warning Linux Bus mapping botch.\n");
		return (busno);

	} else for (i=0; i < MAX_SCG; i++) {
		if (scglocal(scgp)->buscookies[i] == (short)-1) {
			scglocal(scgp)->buscookies[i] = ino;
			return (i);
		}

		if (scglocal(scgp)->buscookies[i] == ino)
			return (i);
	}
	return (0);
}

LOCAL BOOL
sg_mapdev(scgp, f, busp, tgtp, lunp, chanp, inop)
	SCSI	*scgp;
	int	f;
	int	*busp;
	int	*tgtp;
	int	*lunp;
	int	*chanp;
	int	*inop;
{
	struct	sg_id {
		long	l1; /* target | lun << 8 | channel << 16 | low_ino << 24 */
		long	l2; /* Unique id */
	} sg_id;
	int	Chan;
	int	Ino;
	int	Bus;
	int	Target;
	int	Lun;

	if (ioctl(f, SCSI_IOCTL_GET_IDLUN, &sg_id))
		return (FALSE);
	if (scgp->debug > 0) {
		js_fprintf((FILE *)scgp->errfile,
			"l1: 0x%lX l2: 0x%lX\n", sg_id.l1, sg_id.l2);
	}
	if (ioctl(f, SCSI_IOCTL_GET_BUS_NUMBER, &Bus) < 0) {
		Bus = -1;
	}

	Target	= sg_id.l1 & 0xFF;
	Lun	= (sg_id.l1 >> 8) & 0xFF;
	Chan	= (sg_id.l1 >> 16) & 0xFF;
	Ino	= (sg_id.l1 >> 24) & 0xFF;
	if (scgp->debug > 0) {
		js_fprintf((FILE *)scgp->errfile,
			"Bus: %d Target: %d Lun: %d Chan: %d Ino: %d\n",
			Bus, Target, Lun, Chan, Ino);
	}
	*busp = Bus;
	*tgtp = Target;
	*lunp = Lun;
	if (chanp)
		*chanp = Chan;
	if (inop)
		*inop = Ino;
	return (TRUE);
}

#if defined(SG_SET_RESERVED_SIZE) && defined(SG_GET_RESERVED_SIZE)
/*
 * The way Linux does DMA resouce management is a bit curious.
 * It totally deviates from all other OS and forces long ugly code.
 * If we are opening all drivers for a SCSI bus scan operation, we need
 * to set the limit for all open devices.
 * This may use up all kernel memory ... so do the job carefully.
 *
 * A big problem is that SG_SET_RESERVED_SIZE does not return any hint
 * on whether the request did fail. The only way to find if it worked
 * is to use SG_GET_RESERVED_SIZE to read back the current values.
 */
LOCAL long
sg_raisedma(scgp, newmax)
	SCSI	*scgp;
	long	newmax;
{
	register int	b;
	register int	t;
	register int	l;
	register int	f;
		 int	val;
		 int	old;

	/*
	 * First try to raise the DMA limit to a moderate value that
	 * most likely does not use up all kernel memory.
	 */
	val = 126*1024;

	if (val > MAX_DMA_LINUX) {
		for (b=0; b < MAX_SCG; b++) {
			for (t=0; t < MAX_TGT; t++) {
				for (l=0; l < MAX_LUN ; l++) {
					if ((f = SCGO_FILENO(scgp, b, t, l)) < 0)
						continue;
					old = 0;
					if (ioctl(f, SG_GET_RESERVED_SIZE, &old) < 0)
						continue;
					if (val > old)
						ioctl(f, SG_SET_RESERVED_SIZE, &val);
				}
			}
		}
	}

	/*
	 * Now to raise the DMA limit to what we really need.
	 */
	if (newmax > val) {
		val = newmax;
		for (b=0; b < MAX_SCG; b++) {
			for (t=0; t < MAX_TGT; t++) {
				for (l=0; l < MAX_LUN ; l++) {
					if ((f = SCGO_FILENO(scgp, b, t, l)) < 0)
						continue;
					old = 0;
					if (ioctl(f, SG_GET_RESERVED_SIZE, &old) < 0)
						continue;
					if (val > old)
						ioctl(f, SG_SET_RESERVED_SIZE, &val);
				}
			}
		}
	}

	/*
	 * To make sure we did not fail (the ioctl does not report errors)
	 * we need to check the DMA limits. We return the smallest value.
	 */
	for (b=0; b < MAX_SCG; b++) {
		for (t=0; t < MAX_TGT; t++) {
			for (l=0; l < MAX_LUN ; l++) {
				if ((f = SCGO_FILENO(scgp, b, t, l)) < 0)
					continue;
				if (ioctl(f, SG_GET_RESERVED_SIZE, &val) < 0)
					continue;
				if (scgp->debug > 0) {
					js_fprintf((FILE *)scgp->errfile,
						"Target (%d,%d,%d): DMA max %d old max: %ld\n",
						b, t, l, val, newmax);
				}
				if (val < newmax)
					newmax = val;
			}
		}
	}
	return ((long)newmax);
}
#endif

LOCAL long
scgo_maxdma(scgp, amt)
	SCSI	*scgp;
	long	amt;
{
	long maxdma = MAX_DMA_LINUX;

#if defined(SG_SET_RESERVED_SIZE) && defined(SG_GET_RESERVED_SIZE)
	/*
	 * Use the curious new kernel interface found on Linux >= 2.2.10
	 * This interface first appeared in 2.2.6 but it was not working.
	 */
	if (scglocal(scgp)->drvers >= 20134)
		maxdma = sg_raisedma(scgp, amt);
#endif
#ifdef	SG_GET_BUFSIZE
	/*
	 * We assume that all /dev/sg instances use the same
	 * maximum buffer size.
	 */
	maxdma = ioctl(scglocal(scgp)->scgfile, SG_GET_BUFSIZE, 0);
#endif
	if (maxdma < 0) {
#ifdef	USE_PG
		/*
		 * If we only have a Parallel port, just return PP maxdma.
		 */
		if (scglocal(scgp)->pgbus == 0)
			return (pg_maxdma(scgp, amt));
#endif
		if (scglocal(scgp)->scgfile >= 0)
			maxdma = MAX_DMA_LINUX;
	}
#ifdef	USE_PG
	if (scg_scsibus(scgp) == scglocal(scgp)->pgbus)
		return (pg_maxdma(scgp, amt));
	if ((scg_scsibus(scgp) < 0) && (pg_maxdma(scgp, amt) < maxdma))
		return (pg_maxdma(scgp, amt));
#endif
	return (maxdma);
}

LOCAL void *
scgo_getbuf(scgp, amt)
	SCSI	*scgp;
	long	amt;
{
	char	*ret;

	if (scgp->debug > 0) {
		js_fprintf((FILE *)scgp->errfile,
				"scgo_getbuf: %ld bytes\n", amt);
	}
	/*
	 * For performance reason, we allocate pagesize()
	 * bytes before the SCSI buffer to avoid
	 * copying the whole buffer contents when
	 * setting up the /dev/sg data structures.
	 */
	ret = valloc((size_t)(amt+getpagesize()));
	if (ret == NULL)
		return (ret);
	scgp->bufbase = ret;
	ret += getpagesize();
	scglocal(scgp)->SCSIbuf = ret;
	return ((void *)ret);
}

LOCAL void
scgo_freebuf(scgp)
	SCSI	*scgp;
{
	if (scgp->bufbase)
		free(scgp->bufbase);
	scgp->bufbase = NULL;
}

LOCAL BOOL
scgo_havebus(scgp, busno)
	SCSI	*scgp;
	int	busno;
{
	register int	t;
	register int	l;

	if (busno < 0 || busno >= MAX_SCG)
		return (FALSE);

	if (scgp->local == NULL)
		return (FALSE);

	for (t=0; t < MAX_TGT; t++) {
		for (l=0; l < MAX_LUN ; l++)
			if (scglocal(scgp)->scgfiles[busno][t][l] >= 0)
				return (TRUE);
	}
	return (FALSE);
}

LOCAL int
scgo_fileno(scgp, busno, tgt, tlun)
	SCSI	*scgp;
	int	busno;
	int	tgt;
	int	tlun;
{
	if (busno < 0 || busno >= MAX_SCG ||
	    tgt < 0 || tgt >= MAX_TGT ||
	    tlun < 0 || tlun >= MAX_LUN)
		return (-1);

	if (scgp->local == NULL)
		return (-1);

	return ((int)scglocal(scgp)->scgfiles[busno][tgt][tlun]);
}

LOCAL int
scgo_initiator_id(scgp)
	SCSI	*scgp;
{
#ifdef	USE_PG
	if (scg_scsibus(scgp) == scglocal(scgp)->pgbus)
		return (pg_initiator_id(scgp));
#endif
	return (-1);
}

LOCAL int
scgo_isatapi(scgp)
	SCSI	*scgp;
{
#ifdef	USE_PG
	if (scg_scsibus(scgp) == scglocal(scgp)->pgbus)
		return (pg_isatapi(scgp));
#endif

#ifdef	SG_EMULATED_HOST
	{
	int	emulated = FALSE;

	if (ioctl(scgp->fd, SG_EMULATED_HOST, &emulated) >= 0)
		return (emulated != 0);
	}
#endif
	return (-1);
}

LOCAL int
scgo_reset(scgp, what)
	SCSI	*scgp;
	int	what;
{
#ifdef	SG_SCSI_RESET
	int	f = scgp->fd;
	int	func = -1;
#endif
#ifdef	USE_PG
	if (scg_scsibus(scgp) == scglocal(scgp)->pgbus)
		return (pg_reset(scgp, what));
#endif
	/*
	 * Do we have a SCSI reset in the Linux sg driver?
	 */
#ifdef	SG_SCSI_RESET
	/*
	 * Newer Linux sg driver seem to finally implement it...
	 */
#ifdef	SG_SCSI_RESET_NOTHING
	func = SG_SCSI_RESET_NOTHING;
	if (ioctl(f, SG_SCSI_RESET, &func) >= 0) {
		if (what == SCG_RESET_NOP)
			return (0);
#ifdef	SG_SCSI_RESET_DEVICE
		if (what == SCG_RESET_TGT) {
			func = SG_SCSI_RESET_DEVICE;
			if (ioctl(f, SG_SCSI_RESET, &func) >= 0)
				return (0);
		}
#endif
#ifdef	SG_SCSI_RESET_BUS
		if (what == SCG_RESET_BUS) {
			func = SG_SCSI_RESET_BUS;
			if (ioctl(f, SG_SCSI_RESET, &func) >= 0)
				return (0);
		}
#endif
	}
#endif
#endif
	return (-1);
}

LOCAL void
sg_settimeout(f, tmo)
	int	f;
	int	tmo;
{
	tmo *= HZ;
	if (tmo)
		tmo += HZ/2;

	if (ioctl(f, SG_SET_TIMEOUT, &tmo) < 0)
		comerr("Cannot set SG_SET_TIMEOUT.\n");
}

/*
 * Get misaligned int.
 * Needed for all recent processors (sparc/ppc/alpha)
 * because the /dev/sg design forces us to do misaligned
 * reads of integers.
 */
#ifdef	MISALIGN
LOCAL int
sg_getint(ip)
	int	*ip;
{
		 int	ret;
	register char	*cp = (char *)ip;
	register char	*tp = (char *)&ret;
	register int	i;

	for (i = sizeof(int); --i >= 0; )
		*tp++ = *cp++;
	
	return (ret);
}
#define	GETINT(a)	sg_getint(&(a))
#else
#define	GETINT(a)	(a)
#endif

#ifdef	SG_IO
LOCAL int
scgo_send(scgp)
	SCSI		*scgp;
{
	struct scg_cmd	*sp = scgp->scmd;
	int		ret;
	sg_io_hdr_t	sg_io;
	struct timeval	to;

	if (scgp->fd < 0) {
		sp->error = SCG_FATAL;
		sp->ux_errno = EIO;
		return (0);
	}
	if (scglocal(scgp)->isold > 0) {
		return (sg_rwsend(scgp));
	}
	fillbytes((caddr_t)&sg_io, sizeof(sg_io), '\0');

	sg_io.interface_id = 'S';

	if (sp->flags & SCG_RECV_DATA) {
		sg_io.dxfer_direction = SG_DXFER_FROM_DEV;
	} else if (sp->size > 0) {
		sg_io.dxfer_direction = SG_DXFER_TO_DEV;
	} else {
		sg_io.dxfer_direction = SG_DXFER_NONE;
	}
	sg_io.cmd_len = sp->cdb_len;
	if (sp->sense_len > SG_MAX_SENSE)
		sg_io.mx_sb_len = SG_MAX_SENSE;
	else
		sg_io.mx_sb_len = sp->sense_len;
	sg_io.dxfer_len = sp->size;
	sg_io.dxferp = sp->addr;
	sg_io.cmdp = sp->cdb.cmd_cdb;
	sg_io.sbp = sp->u_sense.cmd_sense;
	sg_io.timeout = sp->timeout*1000;
	sg_io.flags |= SG_FLAG_DIRECT_IO;

	ret = ioctl(scgp->fd, SG_IO, &sg_io);
	if (scgp->debug > 0) {
		js_fprintf((FILE *)scgp->errfile,
				"ioctl ret: %d\n", ret);
	}

	if (ret < 0) {
		sp->ux_errno = geterrno();
		/*
		 * Check if SCSI command cound not be send at all.
		 * Linux usually returns EINVAL for an unknoen ioctl.
		 * In case somebody from the Linux kernel team learns that the
		 * corect errno would be ENOTTY, we check for this errno too.
		 */
		if ((sp->ux_errno == ENOTTY || sp->ux_errno == EINVAL)
					&& scglocal(scgp)->isold < 0) {
			scglocal(scgp)->isold = 1;
			return (sg_rwsend(scgp));
		}
		if (sp->ux_errno == ENXIO ||
		    sp->ux_errno == EINVAL || sp->ux_errno == EACCES) {
			return (-1);
		}
	}

	sp->u_scb.cmd_scb[0] = sg_io.status;
	sp->sense_count = sg_io.sb_len_wr;

	if (scgp->debug > 0) {
		js_fprintf((FILE *)scgp->errfile,
				"host_status: %02X driver_status: %02X\n",
				sg_io.host_status, sg_io.driver_status);
	}

	switch (sg_io.host_status) {

	case DID_OK:
			/*
			 * If there is no DMA overrun and there is a
			 * SCSI Status byte != 0 then the SCSI cdb transport
			 * was OK and sp->error must be SCG_NO_ERROR.
			 */
			if ((sg_io.driver_status & DRIVER_SENSE) != 0) {
				if (sp->ux_errno == 0)
					sp->ux_errno = EIO;

				if (sp->u_sense.cmd_sense != 0 &&
				    sp->u_scb.cmd_scb[0] == 0) {
					/*
					 * The Linux SCSI system up to 2.4.xx
					 * trashes the status byte in the
					 * kernel. This is true at least for
					 * ide-scsi emulation. Until this gets
					 * fixed, we need this hack.
					 */
					sp->u_scb.cmd_scb[0] = ST_CHK_COND;
					if (sp->sense_count == 0)
						sp->sense_count = SG_MAX_SENSE;
				}
			}
			break;

	case DID_NO_CONNECT:	/* Arbitration won, retry NO_CONNECT? */
			sp->error = SCG_RETRYABLE;
			break;
	case DID_BAD_TARGET:
			sp->error = SCG_FATAL;
			break;
	
	case DID_TIME_OUT:
			sp->error = SCG_TIMEOUT;
			break;

	default:
		to.tv_sec = sp->timeout;
		to.tv_usec = 500000;
		__scg_times(scgp);

		if (scgp->cmdstop->tv_sec < to.tv_sec ||
		    (scgp->cmdstop->tv_sec == to.tv_sec &&
			scgp->cmdstop->tv_usec < to.tv_usec)) {

			sp->ux_errno = 0;
			sp->error = SCG_TIMEOUT;	/* a timeout */
		} else {
			sp->error = SCG_RETRYABLE;
		}
		break;
	}
	if (sp->error && sp->ux_errno == 0)
		sp->ux_errno = EIO;

	sp->resid = sg_io.resid;
	return (0);
}
#else
#	define	sg_rwsend	scgo_send
#endif

LOCAL int
sg_rwsend(scgp)
	SCSI		*scgp;
{
	int		f = scgp->fd;
	struct scg_cmd	*sp = scgp->scmd;
	struct sg_rq	*sgp;
	struct sg_rq	*sgp2;
	int	i;
	int	pack_len;
	int	reply_len;
	int	amt = sp->cdb_len;
	struct sg_rq {
		struct sg_header	hd;
		unsigned char		buf[MAX_DMA_LINUX+SCG_MAX_CMD];
	} sg_rq;
#ifdef	SG_GET_BUFSIZE		/* We may use a 'sg' version 2 driver	*/
	char	driver_byte;
	char	host_byte;
	char	msg_byte;
	char	status_byte;
#endif

	if (f < 0) {
		sp->error = SCG_FATAL;
		sp->ux_errno = EIO;
		return (0);
	}
#ifdef	USE_PG
	if (scg_scsibus(scgp) == scglocal(scgp)->pgbus)
		return (pg_send(scgp));
#endif
	if (sp->timeout != scgp->deftimeout)
		sg_settimeout(f, sp->timeout);


	sgp2 = sgp = &sg_rq;
	if (sp->addr == scglocal(scgp)->SCSIbuf) {
		sgp = (struct sg_rq *)
			(scglocal(scgp)->SCSIbuf - (sizeof(struct sg_header) + amt));
		sgp2 = (struct sg_rq *)
			(scglocal(scgp)->SCSIbuf - (sizeof(struct sg_header)));
	} else {
		if (scgp->debug > 0) {
			js_fprintf((FILE *)scgp->errfile,
				"DMA addr: 0x%8.8lX size: %d - using copy buffer\n",
				(long)sp->addr, sp->size);
		}
		if (sp->size > (int)(sizeof(sg_rq.buf) - SCG_MAX_CMD)) {

			if (scglocal(scgp)->xbuf == NULL) {
				scglocal(scgp)->xbufsize = scgp->maxbuf;
				scglocal(scgp)->xbuf =
					malloc(scglocal(scgp)->xbufsize +
						SCG_MAX_CMD +
						sizeof(struct sg_header));
				if (scgp->debug > 0) {
					js_fprintf((FILE *)scgp->errfile,
						"Allocted DMA copy buffer, addr: 0x%8.8lX size: %ld\n",
						(long)scglocal(scgp)->xbuf,
						scgp->maxbuf);
				}
			}
			if (scglocal(scgp)->xbuf == NULL ||
				sp->size > scglocal(scgp)->xbufsize) {
				errno = ENOMEM;
				return (-1);
			}
			sgp2 = sgp = (struct sg_rq *)scglocal(scgp)->xbuf;
		}
	}

	/*
	 * This is done to avoid misaligned access of sgp->some_int
	 */
	pack_len = sizeof(struct sg_header) + amt;
	reply_len = sizeof(struct sg_header);
	if (sp->flags & SCG_RECV_DATA) {
		reply_len += sp->size;
	} else {
		pack_len += sp->size;
	}

#ifdef	MISALIGN
	/*
	 * sgp->some_int may be misaligned if (sp->addr == SCSIbuf)
	 * This is no problem on Intel porocessors, however
	 * all other processors don't like it.
	 * sizeof(struct sg_header) + amt is usually not a multiple of
	 * sizeof(int). For this reason, we fill in the values into sg_rq
	 * which is always corectly aligned and then copy it to the real
	 * location if this location differs from sg_rq.
	 * Never read/write directly to sgp->some_int !!!!!
	 */
	fillbytes((caddr_t)&sg_rq, sizeof(struct sg_header), '\0');

	sg_rq.hd.pack_len = pack_len;
	sg_rq.hd.reply_len = reply_len;
	sg_rq.hd.pack_id = scglocal(scgp)->pack_id++;
/*	sg_rq.hd.result = 0;	not needed because of fillbytes() */

	if ((caddr_t)&sg_rq != (caddr_t)sgp)
		movebytes((caddr_t)&sg_rq, (caddr_t)sgp, sizeof(struct sg_header));
#else
	fillbytes((caddr_t)sgp, sizeof(struct sg_header), '\0');

	sgp->hd.pack_len = pack_len;
	sgp->hd.reply_len = reply_len;
	sgp->hd.pack_id = scglocal(scgp)->pack_id++;
/*	sgp->hd.result = 0;	not needed because of fillbytes() */
#endif
	if (amt == 12)
		sgp->hd.twelve_byte = 1;


	for (i = 0; i < amt; i++ ) {
		sgp->buf[i] = sp->cdb.cmd_cdb[i];;
	}
	if (!(sp->flags & SCG_RECV_DATA)) {
		if ((void *)sp->addr != (void *)&sgp->buf[amt])
			movebytes(sp->addr, &sgp->buf[amt], sp->size);
		amt += sp->size;
	}
#ifdef	SG_GET_BUFSIZE
	sgp->hd.want_new  = 1;			/* Order new behaviour	*/
	sgp->hd.cdb_len	  = sp->cdb_len;	/* Set CDB length	*/
	if (sp->sense_len > SG_MAX_SENSE)
		sgp->hd.sense_len = SG_MAX_SENSE;
	else
		sgp->hd.sense_len = sp->sense_len;
#endif
	i = sizeof(struct sg_header) + amt;
	if ((amt = write(f, sgp, i)) < 0) {			/* write */
		sg_settimeout(f, scgp->deftimeout);
		return (-1);
	} else if (amt != i) {
		errmsg("scgo_send(%s) wrote %d bytes (expected %d).\n",
						scgp->cmdname, amt, i);
	}

	if (sp->addr == scglocal(scgp)->SCSIbuf) {
		movebytes(sgp, sgp2, sizeof(struct sg_header));
		sgp = sgp2;
	}
	sgp->hd.sense_buffer[0] = 0;
	if ((amt = read(f, sgp, reply_len)) < 0) {		/* read */
		sg_settimeout(f, scgp->deftimeout);
		return (-1);
	}

	if (sp->flags & SCG_RECV_DATA && ((void *)sgp->buf != (void *)sp->addr)) {
		movebytes(sgp->buf, sp->addr, sp->size);
	}
	sp->ux_errno = GETINT(sgp->hd.result);		/* Unaligned read */
	sp->error = SCG_NO_ERROR;

#ifdef	SG_GET_BUFSIZE
	if (sgp->hd.grant_new) {
		sp->sense_count = sgp->hd.sense_len;
		pack_len    = GETINT(sgp->hd.sg_cmd_status);	/* Unaligned read */
		driver_byte = (pack_len  >> 24) & 0xFF;
		host_byte   = (pack_len  >> 16) & 0xFF;
		msg_byte    = (pack_len  >> 8) & 0xFF;
		status_byte =  pack_len  & 0xFF;

		switch (host_byte) {

		case DID_OK:
				if ((driver_byte & DRIVER_SENSE ||
				    sgp->hd.sense_buffer[0] != 0)
							&& status_byte == 0) {
					/*
					 * The Linux SCSI system up to 2.4.xx
					 * trashes the status byte in the
					 * kernel. This is true at least for
					 * ide-scsi emulation. Until this gets
					 * fixed, we need this hack.
					 */
					status_byte = ST_CHK_COND;
					if (sgp->hd.sense_len == 0)
						sgp->hd.sense_len = SG_MAX_SENSE;
				}
				break;

		case DID_NO_CONNECT:	/* Arbitration won, retry NO_CONNECT? */
				sp->error = SCG_RETRYABLE;
				break;

		case DID_BAD_TARGET:
				sp->error = SCG_FATAL;
				break;
	
		case DID_TIME_OUT:
				sp->error = SCG_TIMEOUT;
				break;

		default:
				sp->error = SCG_RETRYABLE;

				if ((driver_byte & DRIVER_SENSE ||
				    sgp->hd.sense_buffer[0] != 0)
							&& status_byte == 0) {
					status_byte = ST_CHK_COND;
					sp->error = SCG_NO_ERROR;
				}
				if (status_byte != 0 && sgp->hd.sense_len == 0) {
					sgp->hd.sense_len = SG_MAX_SENSE;
					sp->error = SCG_NO_ERROR;
				}
				break;

		}
		if ((host_byte != DID_OK || status_byte != 0) && sp->ux_errno == 0)
			sp->ux_errno = EIO;
		sp->u_scb.cmd_scb[0] = status_byte;
		if (status_byte & ST_CHK_COND) {
			sp->sense_count = sgp->hd.sense_len;
			movebytes(sgp->hd.sense_buffer, sp->u_sense.cmd_sense, sp->sense_count);
		}
	} else
#endif
	{
		if (GETINT(sgp->hd.result) == EBUSY) {	/* Unaligned read */
			struct timeval to;

			to.tv_sec = sp->timeout;
			to.tv_usec = 500000;
			__scg_times(scgp);

			if (scgp->cmdstop->tv_sec < to.tv_sec ||
			    (scgp->cmdstop->tv_sec == to.tv_sec &&
				scgp->cmdstop->tv_usec < to.tv_usec)) {

				sp->ux_errno = 0;
				sp->error = SCG_TIMEOUT;	/* a timeout */
			} else {
				sp->error = SCG_RETRYABLE;	/* may be BUS_BUSY */
			}
		}

		if (sp->flags & SCG_RECV_DATA)
			sp->resid = (sp->size + sizeof(struct sg_header)) - amt;
		else
			sp->resid = 0;	/* sg version1 cannot return DMA resid count */

		if (sgp->hd.sense_buffer[0] != 0) {
			sp->scb.chk = 1;
			sp->sense_count = SG_MAX_SENSE;
			movebytes(sgp->hd.sense_buffer, sp->u_sense.cmd_sense, sp->sense_count);
			if (sp->ux_errno == 0)
				sp->ux_errno = EIO;
		}
	}

	if (scgp->verbose > 0 && scgp->debug > 0) {
#ifdef	SG_GET_BUFSIZE
		js_fprintf((FILE *)scgp->errfile,
				"status: 0x%08X pack_len: %d, reply_len: %d pack_id: %d result: %d wn: %d gn: %d cdb_len: %d sense_len: %d sense[0]: %02X\n",
				GETINT(sgp->hd.sg_cmd_status),
				GETINT(sgp->hd.pack_len),
				GETINT(sgp->hd.reply_len),
				GETINT(sgp->hd.pack_id),
				GETINT(sgp->hd.result),
				sgp->hd.want_new,
				sgp->hd.grant_new,
				sgp->hd.cdb_len,
				sgp->hd.sense_len,	
				sgp->hd.sense_buffer[0]);
#else
		js_fprintf((FILE *)scgp->errfile,
				"pack_len: %d, reply_len: %d pack_id: %d result: %d sense[0]: %02X\n",
				GETINT(sgp->hd.pack_len),
				GETINT(sgp->hd.reply_len),
				GETINT(sgp->hd.pack_id),
				GETINT(sgp->hd.result),
				sgp->hd.sense_buffer[0]);
#endif
#ifdef	DEBUG
		js_fprintf((FILE *)scgp->errfile, "sense: ");
		for (i=0; i < 16; i++)
			js_fprintf((FILE *)scgp->errfile, "%02X ", sgp->hd.sense_buffer[i]);
		js_fprintf((FILE *)scgp->errfile, "\n");
#endif
	}

	if (sp->timeout != scgp->deftimeout)
		sg_settimeout(f, scgp->deftimeout);
	return 0;
}

/*--------------------------------------------------------------------------*/
#ifdef	USE_ATA

#define HOST_EMPTY	0xF
#define HOST_SCSI	0x0
#define HOST_IDE	0x1
#define HOST_USB	0x2
#define HOST_IEEE1389	0x3
#define HOST_PARALLEL	0x4
#define HOST_OTHER	0xE


#define typlocal(p, schillybus)		scglocal(p)->bc[schillybus].typ
#define hostlocal(p, schillybus)	scglocal(p)->bc[schillybus].host
#define buslocal(p, schillybus)		scglocal(p)->bc[schillybus].bus

#define MAX_DMA_ATA (131072)	/* EINVAL (hart) ENOMEM (weich) bei mehr ... */
				/* Bei fehlerhaftem Sense Pointer kommt EFAULT */

LOCAL int scgo_send		__PR((SCSI * scgp));
LOCAL BOOL sg_amapdev		__PR((SCSI * scgp, int f, char *device, int *bus,
					int *target, int *lun));
LOCAL BOOL sg_amapdev_scsi	__PR((SCSI * scgp, int f, int *busp, int *tgtp,
					int *lunp, int *chanp, int *inop));
LOCAL int scgo_aget_first_free_shillybus __PR((SCSI * scgp, int subsystem,
					int host, int bus));
LOCAL int scgo_amerge		__PR((char *path, char *readedlink,
					char *buffer, int buflen));

/* uncomment this when you will get a debug file #define DEBUG */
#ifdef DEBUG
#define LOGFILE "cdrom-linux.log"
#define log(a)	sglog a

LOCAL	void	sglog		__PR((const char *fmt, ...));

#include <vadefs.h>

/* VARARGS1 */
#ifdef	PROTOTYPES
LOCAL void	sglog(const char *fmt, ...)
#else
LOCAL void	error(fmt, va_alist)
	char	*fmt;
	va_dcl
#endif
{
	va_list	args;
	FILE	*f	 = fopen(LOGFILE, "a");

	if (f == NULL) return;

#ifdef	PROTOTYPES
	va_start(args, fmt);
#else
	va_start(args);
#endif
	js_fprintf(f, "%r", fmt, args);
	va_end(args);
	fclose(f);
}
#else
#define log(a)
#endif	/* DEBUG */

LOCAL	int	scan_internal __PR((SCSI * scgp, int *fatal));

LOCAL int
scgo_aopen(scgp, device)
	SCSI	*scgp;
	char	*device;
{
	int	bus = scg_scsibus(scgp);
	int	target = scg_target(scgp);
	int	lun = scg_lun(scgp);

	register int	f;
	register int	b;
	register int	t;
	register int	l;
		 int	nopen = 0;

	if (scgp->overbose)
		error("Warning: Using ATA Packet interface.\n");
	if (scgp->overbose) {
		error("Warning: The related libscg interface code is in pre alpha.\n");
		error("Warning: There may be fatal problems.\n");
	}

device = NULL;
	log(("\n<<<<<<<<<<<<<<<<  LOGGING ON >>>>>>>>>>>>>>>>>\n"));
	if (bus >= MAX_SCHILLY_HOSTS || target >= MAX_TGT || lun >= MAX_LUN) {
		errno = EINVAL;
		if (scgp->errstr)
			js_snprintf(scgp->errstr, SCSI_ERRSTR_SIZE,
				"Illegal value for bus, target or lun '%d,%d,%d'",
				bus, target, lun);

		return (-1);
	}
	/* initialisation */
	if (scgp->local == NULL) {
		scgp->local = malloc(sizeof(struct scg_local));
		if (scgp->local == NULL) {
			return (0);
		}
		scglocal(scgp)->scgfile = -1;
		scglocal(scgp)->SCSIbuf = (char *) -1;
		scglocal(scgp)->pack_id = 5;
		scglocal(scgp)->drvers = -1;
		scglocal(scgp)->isold = -1;
		scglocal(scgp)->xbufsize = 0L;
		scglocal(scgp)->xbuf = NULL;

		for (b = 0; b < MAX_SCHILLY_HOSTS; b++) {
			typlocal(scgp, b) = HOST_EMPTY;
			for (t = 0; t < MAX_TGT; t++) {
				for (l = 0; l < MAX_LUN; l++)
					scglocal(scgp)->scgfiles[b][t][l] = (short) -1;
			}
		}
	}
	/* initialisation done */

	/* if not scanning */
	if ((device != NULL && *device != '\0') || (bus == -2 && target == -2))
		goto openbydev;

	if (scan_internal(scgp, &nopen)) {
		if (scgp->errstr)
			js_printf(scgp->errstr, "INFO: scan_internal(...) failed");
		return (-1);
	}
	return (nopen);

openbydev:
	if (scgp->debug > 3) {
		js_fprintf((FILE *) scgp->errfile, "INFO: do scgo_open openbydev");
	}
	if (device != NULL && *device != '\0') {
		int	schilly_bus,
			starget,
			slun;

		f = open(device, O_RDONLY | O_NONBLOCK);

		if (f < 0) {
			if (scgp->errstr)
				js_snprintf(scgp->errstr, SCSI_ERRSTR_SIZE,
					"Cannot open '%s'", device);
			return (0);
		}
		if (sg_amapdev(scgp, f, device, &schilly_bus, &starget, &slun)) {
			scg_settarget(scgp, schilly_bus, starget, slun);
			return (++nopen);
		}
	}
	return (nopen);
}

LOCAL int
scan_internal(scgp, nopen)
	SCSI	*scgp;
	int	*nopen;
{
	int	i,
		f;
	int	schilly_bus,
		target,
		lun;
	char	device[128];
	BOOL	DEVFS = TRUE;	/* try always with devfs
				   unfortunatelly the solution with test of existing
				   of '/dev/.devfsd' don't work, because it root.root 700
				   and i don't like run suid root
				 */

	if (DEVFS) {
		for (i = 0;; i++) {
			sprintf(device, "/dev/cdroms/cdrom%i", i);
			if ((f = open(device, O_RDONLY | O_NONBLOCK)) < 0) {
				if (errno != ENOENT && errno != ENXIO && errno != ENODEV && errno != EACCES) {
					if (scgp->debug > 4) {
						js_fprintf((FILE *) scgp->errfile,
						"try open(%s) return %i, errno %i, cancel\n", device, f, errno);
					}
					return (-2);
				} else if (errno == ENOENT || errno == ENODEV) {
					if (scgp->debug > 4) {
						js_fprintf((FILE *) scgp->errfile,
						"try open(%s) return %i, errno %i\n", device, f, errno);
					}
					if (0 == i) {
						DEVFS = FALSE;
						if (scgp->debug > 4) {
							js_fprintf((FILE *) scgp->errfile,
							"DEVFS not detected, continue with old dev\n");
						}
					}
					break;
				}
				if (scgp->debug > 4) {
					if (errno == EACCES) {
						js_fprintf((FILE *) scgp->errfile,
						"errno (EACCESS), you have'nt right for device, referenced by %s \n", device);
					}
					js_fprintf((FILE *) scgp->errfile,
					"try open(%s) return %i, errno %i, try next cdrom\n", device, f, errno);
				}
			} else {
				if (scgp->debug > 4) {
					js_fprintf((FILE *) scgp->errfile,
					"try open(%s) return %i errno %i do sg_mapdev(...)\n", device, f, errno);
				}
				if (sg_amapdev(scgp, f, device, &schilly_bus, &target, &lun)) {
					(++(*nopen));
				} else {
					close(f);
				}
			}
		}
	}
	if (!DEVFS) {
		/* for /dev/sr0 - /dev/sr? */
		for (i = 0;; i++) {
			sprintf(device, "/dev/sr%i", i);
			if ((f = open(device, O_RDONLY | O_NONBLOCK)) < 0) {
				if (errno != ENOENT && errno != ENXIO && errno != ENODEV && errno != EACCES) {
					if (scgp->debug > 4) {
						js_fprintf((FILE *) scgp->errfile,
						"try open(%s) return %i, errno %i, cancel\n", device, f, errno);
					}
					return (-2);
				} else if (errno == ENOENT || errno == ENODEV) {
					break;
				}
			} else {
				if (sg_amapdev(scgp, f, device, &schilly_bus, &target, &lun)) {
					(++(*nopen));
				} else {
					close(f);
				}
			}
		}

		/* for /dev/hda - /dev/hdh */
		for (i = 'a'; i <= 'h'; i++) {
			sprintf(device, "/dev/hd%c", i);
			if ((f = open(device, O_RDONLY | O_NONBLOCK)) < 0) {
				if (errno != ENOENT && errno != ENXIO && errno != EACCES) {
					if (scgp->debug > 4) {
						js_fprintf((FILE *) scgp->errfile, "try open(%s) return %i, errno %i, cancel\n", device, f, errno);
					}
					return (-2);
				} else if (errno == ENOENT || errno == ENODEV) {
					break;
				}
			} else {
				/* ugly hack, make better, when you can. Alex */
				if (0 > ioctl(f, CDROM_DRIVE_STATUS, CDSL_CURRENT)) {
					if (scgp->debug > 4) {
						js_fprintf((FILE *) scgp->errfile, "%s is not a cdrom, skip\n", device);
					}
					close(f);
				} else if (sg_amapdev(scgp, f, device, &schilly_bus, &target, &lun)) {
					(++(*nopen));
				} else {
					close(f);
				}
			}
		}
	}
	return (0);
}

LOCAL int
scgo_aclose(scgp)
	SCSI	*scgp;
{
	register int	f;
	register int	h;
	register int	t;
	register int	l;

	if (scgp->local == NULL)
		return (-1);

	for (h = 0; h < MAX_SCHILLY_HOSTS; h++) {
		typlocal(scgp, h) = (HOST_EMPTY);
		for (t = 0; t < MAX_TGT; t++) {
			for (l = 0; l < MAX_LUN; l++) {
				f = scglocal(scgp)->scgfiles[h][t][l];
				if (f >= 0)
					close(f);
				scglocal(scgp)->scgfiles[h][t][l] = (short) -1;
			}
		}
	}

	if (scglocal(scgp)->xbuf != NULL) {
		free(scglocal(scgp)->xbuf);
		scglocal(scgp)->xbufsize = 0L;
		scglocal(scgp)->xbuf = NULL;
	}
	log(("<<<<<<<<<<<<<<<<  LOGGING OFF >>>>>>>>>>>>>>>>>\n\n"));
	return (0);
}

LOCAL int
scgo_aget_first_free_shillybus(scgp, subsystem, host, bus)
	SCSI	*scgp;
	int	subsystem;
	int	host;
	int	bus;
{
	int	first_free_schilly_bus;

	for (first_free_schilly_bus = 0;
			first_free_schilly_bus < MAX_SCHILLY_HOSTS;
						first_free_schilly_bus++) {

		if (typlocal(scgp, first_free_schilly_bus) == HOST_EMPTY ||
		   (typlocal(scgp, first_free_schilly_bus) == subsystem &&
		    hostlocal(scgp, first_free_schilly_bus) == host &&
		    buslocal(scgp, first_free_schilly_bus) == bus))
			break;
	}

	if (first_free_schilly_bus >= MAX_SCHILLY_HOSTS) {
		errmsgno(EX_BAD, "ERROR: in scgo_get_first_free_shillybus(...). Too much CDROMs, more as %i",
			MAX_SCHILLY_HOSTS);
		errmsgno(EX_BAD, "Increase MAX_SCHILLY_HOSTS in cdrom-linux.c and recompile again!");
		return (-1);
	}
	return (first_free_schilly_bus);
}

LOCAL int
scgo_amerge(path, readedlink, buffer, buflen)
	char	*path;
	char	*readedlink;
	char	*buffer;
	int	buflen;
{
	char	*aa;

#define TOKEN_ARRAY		20
#define LAST_CHAR(x)		(x)[strlen((x))-1]
#define ONE_CHAR_BACK(x)	(x)[strlen((x))-1] = '\0'
	char	*ppa[TOKEN_ARRAY];
	char	*pa;

	int	i;
	int	len;
	char	seps[] = "/";
	char	*last_slash;

	if (!path || !readedlink || !buffer)
		return -(EINVAL);

	if ('/' == readedlink[0]) {
		aa = (char *) malloc(strlen(readedlink) + 1);
		if (!aa)
			return -(ENOMEM);

		strcpy(aa, readedlink);
	} else {
		aa = (char *) malloc(strlen(path) + strlen(readedlink) + 1);
		if (!aa)
			return -(ENOMEM);

		strcpy(aa, path);
		if (LAST_CHAR(aa) == '/') {
			ONE_CHAR_BACK(aa);
		}
		last_slash = strrchr(aa, '/');
		if (last_slash == NULL)
			strcpy(aa, "/");
		else
			*(++last_slash) = '\0';
		strcat(aa, readedlink);
	}
	memset(ppa, 0x00, sizeof(ppa));

	for (i = 0, pa = strtok(aa, seps);
		i < TOKEN_ARRAY && pa != NULL;
		++i, pa = strtok(NULL, seps)) {
		ppa[i] = pa;
	}

	if (i == TOKEN_ARRAY) {
		free(aa);
		return -(ENOMEM);
	}
	for (i = 0; i < TOKEN_ARRAY && ppa[i]; i++) {
		if (strcmp(ppa[i], "..") == 0) {
			ppa[i] = NULL;
			if (i > 1)
				ppa[i - 1] = NULL;
		}
	}

	/* dry run */
	len = 0;
	for (i = 0; i < TOKEN_ARRAY; i++) {
		if (ppa[i]) {
			len += 1;
			len += strlen(ppa[i]);
		}
	}
	if (0 == len)
		len = 1;

	if (len + 1 <= buflen) {
		strcpy(buffer, "");
		for (i = 0; i < TOKEN_ARRAY; i++) {
			if (ppa[i]) {
				strcat(buffer, "/");
				strcat(buffer, ppa[i]);
			}
		}

		if (strlen(buffer) == 0)
			strcpy(buffer, "/");
	}
	free(aa);

	return (len + 1);
}

/*
 *	/dev/cdroms/cdrom0	first CD-ROM
 *	/dev/cdroms/cdrom1	second CD-ROM
 *
 *
 *	SCSI Devices
 *
 *	To uniquely identify any SCSI device requires the following information:
 *
 *	controller	(host adapter)
 *	bus		(SCSI channel)
 *	target		(SCSI ID)
 *	unit		(Logical Unit Number)
 *
 *	All SCSI devices are placed under /dev/scsi (assuming devfs is mounted on /dev).
 *	Hence, a SCSI device with the following parameters:
 *		c=1,b=2,t=3,u=4 would appear as:
 *
 *		/dev/scsi/host1/bus2/target3/lun4	device directory
 *
 *	Inside this directory, a number of device entries may be created,
 *	depending on which SCSI device-type drivers were installed.
 *
 *	See the section on the disc naming scheme to see what entries
 *	the SCSI disc driver creates.
 *
 *	See the section on the tape naming scheme to see what entries
 *	the SCSI tape driver creates.
 *
 *	The SCSI CD-ROM driver creates:  cd
 *	The SCSI generic driver creates: generic
 *
 *	IDE Devices
 *
 *	To uniquely identify any IDE device requires the following information:
 *
 *	controller
 *	bus		(0/1 aka. primary/secondary)
 *	target		(0/1 aka. master/slave)
 *	unit
 *
 *	All IDE devices are placed under /dev/ide, and uses a similar
 *	naming scheme to the SCSI subsystem.
 *
 *
 *	Example /dev/cdroms/cdrom0 ->  /dev/scsi/host1/bus2/target3/lun4/cd
 *	Example /dev/cdroms/cdrom1 ->  /dev/ide/host1/bus0/target1/lun4/cd
 *
 */
LOCAL BOOL
sg_amapdev(scgp, f, device, schillybus, target, lun)
	SCSI	*scgp;
	int	f;
	char	*device;
	int	*schillybus;
	int	*target;
	int	*lun;
{
	struct host {
		char	host[4];
		char	host_no;
	};
	struct bus {
		char	bus[3];
		char	bus_no;
	};
	struct target {
		char	target[6];
		char	target_no;
	};
	struct lun {
		char	lun[3];
		char	lun_no;
	};

	int	h,
		b,
		t,
		l;

#define TOKEN_DEV		"dev"
#define TOKEN_SUBSYSTEM_SCSI	"scsi"
#define TOKEN_SUBSYSTEM_IDE	"ide"
#define TOKEN_HOST		"host"
#define TOKEN_BUS		"bus"
#define TOKEN_TARGET		"target"
#define TOKEN_LUN		"lun"
#define TOKEN_CD		"cd"

#define ID_TOKEN_DEV		0
#define ID_TOKEN_SUBSYSTEM	1
#define ID_TOKEN_HOST		2
#define ID_TOKEN_BUS		3
#define ID_TOKEN_TARGET		4
#define ID_TOKEN_LUN		5
#define ID_TOKEN_CD		6
#define ID_TOKEN_LAST		ID_TOKEN_CD
#define ID_TOKEN_MAX		ID_TOKEN_LAST + 2
#define CHARTOINT(x)		(abs(atoi(&x)))

	char		*token[ID_TOKEN_MAX],
			*seps = "/";
	int		i,
			result;
	struct stat	buf;

#ifndef MAX_PATH
#define MAX_PATH 260
#endif
#define LOCAL_MAX_PATH MAX_PATH
	char		tmp[LOCAL_MAX_PATH],
			tmp1[LOCAL_MAX_PATH];
	int		first_free_schilly_bus;
	int		subsystem = HOST_EMPTY;

	/* old DEV */
	typedef struct {
		char		prefix[2];
		char		device;
	} old_dev;
	/* strtok need char* instead of const char* */
	result = stat(device, &buf);
	if (result || !S_ISBLK(buf.st_mode))
		return (FALSE);

	result = lstat(device, &buf);
	if (!result && S_ISLNK(buf.st_mode)) {
		result = readlink(device, tmp, LOCAL_MAX_PATH);
		if (result > 0 && result < LOCAL_MAX_PATH) {
			tmp[result] = '\0';

			result = scgo_amerge(device, tmp, tmp1, LOCAL_MAX_PATH);
			if (result > 0 && result < LOCAL_MAX_PATH) {
				tmp1[result] = '\0';
				strcpy(tmp, tmp1);
			} else {
				errmsgno(EX_BAD, "ERROR: by link merging! base %s link %s, result of merging %i\n",
					device, tmp, result);
				return (FALSE);
			}
		} else {
			errmsgno(EX_BAD, "ERROR: by link reading! link %s, result of readlink %i\n",
				device, result);
			return (FALSE);
		}
	} else {
		strncpy(tmp, device, sizeof(tmp));
	}
	if (scgp->debug > 3) {
		js_fprintf((FILE *) scgp->errfile, "INFO: %s -> %s\n", device, tmp);
	}
	memset(token, 0x00, sizeof(token));
	i = 0;
	token[i] = strtok(tmp, seps);
	while (token[i] != NULL && (++i) && i < ID_TOKEN_MAX) {
		token[i] = strtok(NULL, seps);
	}

	if (i == ID_TOKEN_MAX ||
		!(token[ID_TOKEN_DEV]) ||
		strcmp(token[ID_TOKEN_DEV], TOKEN_DEV)) {

		errmsgno(EX_BAD, "ERROR: unknow format\n");
		errmsgno(EX_BAD, "EXAMPLE: /dev/scsi/host1/bus2/target3/lun4/cd\n");
		errmsgno(EX_BAD, "EXAMPLE: /dev/ide/host0/bus0/target1/lun0/cd\n");
		errmsgno(EX_BAD, "EXAMPLE: /dev/hda or /dev/sr0\n");
		return (FALSE);
	}
	if (!(strcmp(token[ID_TOKEN_SUBSYSTEM], TOKEN_SUBSYSTEM_SCSI))
		|| !(strcmp(token[ID_TOKEN_SUBSYSTEM], TOKEN_SUBSYSTEM_IDE))) {
		h = CHARTOINT(((struct host *) token[ID_TOKEN_HOST])->host_no);
		b = CHARTOINT(((struct bus *) token[ID_TOKEN_BUS])->bus_no);
		t = CHARTOINT(((struct target *) token[ID_TOKEN_TARGET])->target_no);
		l = CHARTOINT(((struct lun *) token[ID_TOKEN_LUN])->lun_no);
#ifdef PARANOID
		if (strncmp(token[ID_TOKEN_HOST], TOKEN_HOST, strlen(TOKEN_HOST))) {
			log(("ERROR: invalid host specified\n"));
			return (FALSE);
		}
		if (strncmp(token[ID_TOKEN_BUS], TOKEN_BUS, strlen(TOKEN_BUS))) {
			log(("ERROR: invalid bus specified\n"));
			return (FALSE);
		}
		if (strncmp(token[ID_TOKEN_TARGET], TOKEN_TARGET, strlen(TOKEN_TARGET))) {
			log(("ERROR: invalid target specified\n"));
			return (FALSE);
		}
		if (strncmp(token[ID_TOKEN_LUN], TOKEN_LUN, strlen(TOKEN_LUN))) {
			log(("ERROR: invalid lun specified\n"));
			return (FALSE);
		}
		if (!(strcmp(token[ID_TOKEN_SUBSYSTEM], TOKEN_SUBSYSTEM_IDE))) {
			if (b > 1 || t > 1) {
				log(("ERROR: invalid bus or target for IDE specified\n"));
				return (FALSE);
			}
		}
#endif	/* PARANOID */

		if (!(strcmp(token[ID_TOKEN_SUBSYSTEM], TOKEN_SUBSYSTEM_IDE))) {
			subsystem = HOST_IDE;
		} else if (!(strcmp(token[ID_TOKEN_SUBSYSTEM], TOKEN_SUBSYSTEM_SCSI))) {
			subsystem = HOST_SCSI;
		} else {
			subsystem = HOST_OTHER;
		}
	} else if (!token[ID_TOKEN_HOST] &&
		strlen(token[ID_TOKEN_SUBSYSTEM]) == sizeof(old_dev)) {
		char	j;

		old_dev	*pDev = (old_dev *) token[ID_TOKEN_SUBSYSTEM];

		if (!strncmp(pDev->prefix, "hd", 2)) {
			j = pDev->device - ('a');

			subsystem = HOST_IDE;
			h = j / 4;
			b = (j % 4) / 2;
			t = (j % 4) % 2;
			l = 0;
		} else if (!strncmp(pDev->prefix, "sr", 2)) {
/*
			if(pDev->device >= '0' && pDev->device <= '9')
				j = pDev->device - ('0');
			else
				j = pDev->device - ('a');


			h = j / 4;
			b = (j % 4) / 2;
			t = (j % 4) % 2;
			l = 0;
*/
			/* other solution, with ioctl */
			int	Chan,
				Ino,
				Bus,
				Target,
				Lun;

			subsystem = HOST_SCSI;
			sg_amapdev_scsi(scgp, f, &Bus, &Target, &Lun, &Chan, &Ino);

			/* For old kernels try to make the best guess. */
			/*
				int	n;
				Ino |= Chan << 8;
				n = sg_mapbus(scgp, Bus, Ino);
				if (Bus == -1) {
					Bus = n;
					if (scgp->debug > 0) {
						js_fprintf((FILE *)scgp->errfile,
							"SCSI Bus: %d (mapped from %d)\n",
							Bus, Ino);
					}
				} 
				It is me too high ;-()
			*/
			h = Ino;
			b = Chan;
			t = Target;
			l = Lun;
		} else {
			errmsgno(EX_BAD, "ERROR: unknow subsystem (%s) in (%s)\n",
				token[ID_TOKEN_SUBSYSTEM], device);
			return (FALSE);
		}
	} else {
		errmsgno(EX_BAD, "ERROR: unknow subsystem (%s) in (%s)\n",
			token[ID_TOKEN_SUBSYSTEM], device);
		return (FALSE);
	}

	if (scgp->verbose)
		js_printf(scgp->errstr, "INFO: subsystem %s: h %i, b %i, t %i, l %i\n",
			token[ID_TOKEN_SUBSYSTEM], h, b, t, l);

	first_free_schilly_bus = scgo_aget_first_free_shillybus(scgp, subsystem, h, b);
	if (-1 == first_free_schilly_bus) {
		return (FALSE);
	}
	if (scglocal(scgp)->scgfiles[first_free_schilly_bus][t][l] != (-1)) {
		errmsgno(EX_BAD, "ERROR: this cdrom is already mapped %s(%d,%d,%d)\n",
			device, first_free_schilly_bus, t, l);
		return (FALSE);
	} else {
		scglocal(scgp)->scgfiles[first_free_schilly_bus][t][l] = f;
		typlocal(scgp, first_free_schilly_bus) = subsystem;
		hostlocal(scgp, first_free_schilly_bus) = h;
		buslocal(scgp, first_free_schilly_bus) = b;
		*schillybus = first_free_schilly_bus;
		*target = t;
		*lun = l;

		if (scgp->debug > 1) {
			js_fprintf((FILE *) scgp->errfile,
				"INFO: /dev/%s, (host%d/bus%d/target%d/lun%d) will be mapped on the schilly bus No %d (%d,%d,%d)\n",
				token[ID_TOKEN_SUBSYSTEM], h, b, t, l, first_free_schilly_bus, first_free_schilly_bus, t, l);
		}
	}
	return (TRUE);
}

LOCAL BOOL
sg_amapdev_scsi(scgp, f, busp, tgtp, lunp, chanp, inop)
	SCSI	*scgp;
	int	f;
	int	*busp;
	int	*tgtp;
	int	*lunp;
	int	*chanp;
	int	*inop;
{
	struct sg_id {
		long	l1;	/* target | lun << 8 | channel << 16 | low_ino << 24 */
		long	l2;	/* Unique id */
	} sg_id;
	int	Chan;
	int	Ino;
	int	Bus;
	int	Target;
	int	Lun;

	if (ioctl(f, SCSI_IOCTL_GET_IDLUN, &sg_id))
		return (FALSE);

	if (scgp->debug > 0) {
		js_fprintf((FILE *) scgp->errfile, "INFO: l1: 0x%lX l2: 0x%lX\n", sg_id.l1, sg_id.l2);
	}
	if (ioctl(f, SCSI_IOCTL_GET_BUS_NUMBER, &Bus) < 0) {
		Bus = -1;
	}
	Target = sg_id.l1 & 0xFF;
	Lun = (sg_id.l1 >> 8) & 0xFF;
	Chan = (sg_id.l1 >> 16) & 0xFF;
	Ino = (sg_id.l1 >> 24) & 0xFF;
	if (scgp->debug > 0) {
		js_fprintf((FILE *) scgp->errfile, "INFO: Bus: %d Target: %d Lun: %d Chan: %d Ino: %d\n",
			Bus, Target, Lun, Chan, Ino);
	}
	*busp = Bus;
	*tgtp = Target;
	*lunp = Lun;
	if (chanp)
		*chanp = Chan;
	if (inop)
		*inop = Ino;
	return (TRUE);
}

LOCAL long
scgo_amaxdma(scgp, amt)
	SCSI	*scgp;
	long	amt;
{
	/*
	 * EINVAL (hart) ENOMEM (weich) bei mehr ...
	 * Bei fehlerhaftem Sense Pointer kommt EFAULT
	 */
	return (MAX_DMA_ATA);
}

LOCAL BOOL
scgo_ahavebus(scgp, busno)
	SCSI	*scgp;
	int	busno;
{
	register int	t;
	register int	l;

	if (busno < 0 || busno >= MAX_SCHILLY_HOSTS)
		return (FALSE);

	if (scgp->local == NULL)
		return (FALSE);

	for (t = 0; t < MAX_TGT; t++) {
		for (l = 0; l < MAX_LUN; l++)
			if (scglocal(scgp)->scgfiles[busno][t][l] >= 0)
				return (TRUE);
	}
	return (FALSE);
}

LOCAL int
scgo_afileno(scgp, busno, tgt, tlun)
	SCSI	*scgp;
	int	busno;
	int	tgt;
	int	tlun;
{
	if (busno < 0 || busno >= MAX_SCHILLY_HOSTS ||
		tgt < 0 || tgt >= MAX_TGT ||
		tlun < 0 || tlun >= MAX_LUN)
		return (-1);

	if (scgp->local == NULL)
		return (-1);

	return ((int) scglocal(scgp)->scgfiles[busno][tgt][tlun]);
}

LOCAL int
scgo_ainitiator_id(scgp)
	SCSI	*scgp;
{
	js_printf(scgp->errstr, "NOT IMPELEMENTED: scgo_initiator_id\n");
	return (-1);
}

LOCAL int
scgo_aisatapi(scgp)
	SCSI	*scgp;
{
	int schillybus = scgp->addr.scsibus;
	int typ = typlocal(scgp, schillybus);
	if(typ == HOST_EMPTY)
		return (-1);
	if(typ != HOST_SCSI)
		return (1);
	else
		return (0);
}

LOCAL int
scgo_areset(scgp, what)
	SCSI	*scgp;
	int	what;
{
	if (what == SCG_RESET_NOP)
		return (0);

	if (what == SCG_RESET_TGT || what == SCG_RESET_BUS)
		return (ioctl(what, CDROMRESET));

	return (-1);
}

LOCAL int
scgo_asend(scgp)
	SCSI	*scgp;
{
	struct scg_cmd	*sp = scgp->scmd;
	int		ret,
			i;
	struct cdrom_generic_command sg_cgc;
	struct request_sense sense_cgc;

#ifdef DEBUG
	char		tmp_send[340],
			tmp_read[340],
			tmp_sense[340],
			tmp1[30];
	int		j;
	char		*p;
#endif

	if (scgp->fd < 0) {
		sp->error = SCG_FATAL;
		sp->ux_errno = EIO;
		return (0);
	}
	if (sp->cdb_len > CDROM_PACKET_SIZE) {
		sp->error = SCG_FATAL;
		sp->ux_errno = EIO;
		return (0);
	}
	/* initialize */
	fillbytes((caddr_t) & sg_cgc, sizeof(sg_cgc), '\0');
	fillbytes((caddr_t) & sense_cgc, sizeof(sense_cgc), '\0');

	if (sp->flags & SCG_RECV_DATA) {
		sg_cgc.data_direction = CGC_DATA_READ;
	} else if (sp->size > 0) {
		sg_cgc.data_direction = CGC_DATA_WRITE;
	} else {
		sg_cgc.data_direction = CGC_DATA_NONE;
	}
#if LINUX_VERSION_CODE >= 0x020403
	if (sp->flags & SCG_SILENT) {
		sg_cgc.quiet = 1;
	}
#endif
	for (i = 0; i < sp->cdb_len; i++) {
		sg_cgc.cmd[i] = sp->cdb.cmd_cdb[i];
	}

	sg_cgc.buflen = sp->size;
	sg_cgc.buffer = sp->addr;

	if (sp->sense_len > sizeof(sense_cgc))
		sense_cgc.add_sense_len = sizeof(sense_cgc) - 8;
	else
		sense_cgc.add_sense_len = sp->sense_len - 8;

	sg_cgc.sense = &sense_cgc;
#if LINUX_VERSION_CODE >= 0x020403
	sg_cgc.timeout = sp->timeout * 1000;
#endif
#ifdef DEBUG
	strcpy(tmp_send, "send cmd:\n");
	for (j = 0; j < sp->cdb_len; j++) {
		sprintf(tmp1, " %02X", sp->cdb.cmd_cdb[j]);
		strcat(tmp_send, tmp1);
	}
	strcat(tmp_send, "\n");

	if (sg_cgc.data_direction == CGC_DATA_WRITE) {
		int	z;

		sprintf(tmp1, "data_write: %i bytes\n", sp->size);
		strcat(tmp_send, tmp1);
		for (j = 0, z = 1; j < 80 && j < sp->size; j++, z++) {
			if (z > 16) {
				z = 1;
				strcat(tmp_send, "\n");
			}
			sprintf(tmp1, " %02X", (unsigned char) (sp->addr[j]));
			strcat(tmp_send, tmp1);
		}
		strcat(tmp_send, "\n");

		if (sp->size > 80) {
			strcat(tmp_send, "...\n");
		}
	}
#endif	/* DEBUG */
	if ((ret = ioctl(scgp->fd, CDROM_SEND_PACKET, &sg_cgc)) < 0)
		sp->ux_errno = geterrno();

	if (ret < 0 && scgp->debug > 4) {
		js_fprintf((FILE *) scgp->errfile,
			"ioctl(CDROM_SEND_PACKET) ret: %d\n", ret);
	}
	/*
	 * copy scsi data back
	 */
	if (sp->flags & SCG_RECV_DATA && ((void *) sp->addr != (void *) sg_cgc.buffer)) {
		memcpy(sp->addr, sg_cgc.buffer, (sp->size < sg_cgc.buflen) ? sp->size : sg_cgc.buflen);
		if (sg_cgc.buflen > sp->size)
			sp->resid = sg_cgc.buflen - sp->size;
	}
	sp->error = SCG_NO_ERROR;
#ifdef DEBUG
	if (ret < 0) {
		switch (sp->ux_errno) {
		case ENOTTY:
			p = "ENOTTY";
			break;
		case EINVAL:
			p = "EINVAL";
			break;
		case ENXIO:
			p = "ENXIO";
			break;
		case EACCES:
			p = "EACCES";
			break;
		case EIO:
			p = "EIO";
			break;
		case ENOMEDIUM:
			p = "ENOMEDIUM";
			break;
		case EDRIVE_CANT_DO_THIS:
			p = "EDRIVE_CANT_DO_THIS";
			break;
		default:
			p = "UNKNOW";
		};
		log(("%s", tmp_send));
		log(("ERROR: returns %i errno %i(%s)\n", ret, sp->ux_errno, p));
	}
#endif	/* DEBUG */
	if (ret < 0) {
		/*
		 * Check if SCSI command cound not be send at all.
		 * Linux usually returns EINVAL for an unknoen ioctl.
		 * In case somebody from the Linux kernel team learns that the
		 * corect errno would be ENOTTY, we check for this errno too.
		*/
		if ((sp->ux_errno == ENOTTY || sp->ux_errno == EINVAL)) {
			return (-1);
		}
		if (sp->ux_errno == ENXIO ||
			sp->ux_errno == EINVAL || sp->ux_errno == EACCES) {
			return (-1);
		}
	} else if (ret == 0) {
#ifdef DEBUG
		if (sg_cgc.data_direction == CGC_DATA_READ) {
			int	z;

			sprintf(tmp_read, "data_read: %i bytes\n", sp->size);
			for (j = 0, z = 1; j < 80 && j < sp->size; j++, z++) {
				if (z > 16) {
					z = 1;
					strcat(tmp_read, "\n");
				}
				sprintf(tmp1, " %02X", (unsigned char) (sp->addr[j]));
				strcat(tmp_read, tmp1);
			}
			strcat(tmp_read, "\n");
			if (sp->size > 80) {
				strcat(tmp_read, "...\n");
			}
		}
#endif	/* DEBUG */
	}
	/*
	 * copy sense back
	 */
	if (ret < 0 && sg_cgc.sense->error_code) {
		sp->sense_count = sense_cgc.add_sense_len + 8;
#ifdef DEBUG
		sprintf(tmp_sense, "sense_data: Laenge %i\n", sp->sense_count);
		for (j = 0; j < sp->sense_count; j++) {
			sprintf(tmp1, " %02X", (((unsigned char *) (&sense_cgc))[j]));
			strcat(tmp_sense, tmp1);
		}
		log(("%s\n", tmp_sense));

		sprintf(tmp_sense, "sense_data: error code 0x%02X, sense key 0x%02X,"
			" additional length %i, ASC 0x%02X, ASCQ 0x%02X\n",
			sg_cgc.sense->error_code, sg_cgc.sense->sense_key,
			sg_cgc.sense->add_sense_len, sg_cgc.sense->asc,
			sg_cgc.sense->ascq);

		log(("%s\n", tmp_sense));
#endif	/* DEBUG */
		memcpy(sp->u_sense.cmd_sense, /* (caddr_t) */ &sense_cgc, SCG_MAX_SENSE);
		sp->u_scb.cmd_scb[0] = ST_CHK_COND;

		switch (sg_cgc.sense->sense_key) {
		case SC_UNIT_ATTENTION:
		case SC_NOT_READY:
			sp->error = SCG_RETRYABLE;	/* may be BUS_BUSY */
			sp->u_scb.cmd_scb[0] |= ST_BUSY;
			break;
		case SC_ILLEGAL_REQUEST:
			break;
		default:
			break;
		}
	} else {
		sp->u_scb.cmd_scb[0] = 0x00;
	}

	sp->resid = 0;
	return (0);
}
#endif	/* USE_ATA */
