/*
 *  This file is part of the KDE libraries
 *  Copyright (c) 2001 Michael Goffioul <goffioul@imec.be>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License version 2 as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 **/

#include "printwrapper.h"

#include <unistd.h>
#include <signal.h>

#include <qstring.h>
#include <qstringlist.h>
#include <stdlib.h>
#include <kmessagebox.h>
#include <qfile.h>
#include <qtimer.h>
#include <qregexp.h>

#include <kapplication.h>
#include <kcmdlineargs.h>
#include <klocale.h>
#include <kstandarddirs.h>
#include <kio/netaccess.h>
#include <kurl.h>
#include <kdebug.h>

#include <kprinter.h>
#include <kdeprint/kmmanager.h>
#include <kdeprint/kmprinter.h>
#include <kdeprint/kprintdialog.h>

void signal_handler(int);
QString tempFile;
bool fromStdin = false;
char job_output = 0;	// 0: dialog, 1: console, 2: none

void showmsgdialog(const QString& msg, int type = 0)
{
	switch (type)
	{
	   case 0: KMessageBox::information(NULL,msg,i18n("Print information")); break;
	   case 1: KMessageBox::sorry(NULL,msg,i18n("Print warning")); break;
	   case 2: KMessageBox::error(NULL,msg,i18n("Print error")); break;
	}
}

void showmsgconsole(const QString& msg, int type = 0)
{
	QString	errmsg = QString::fromLatin1("%1 : ").arg((type == 0 ? i18n("Print info") : (type == 1 ? i18n("Print warning") : i18n("Print error"))));
	kdDebug() << errmsg << msg << endl;
}

void showmsg(const QString& msg, int type = 0)
{
	switch (job_output) {
	   case 0: showmsgdialog(msg,type); break;
	   case 1: showmsgconsole(msg,type); break;
	   default: break;
	}
}

void errormsg(const QString& msg)
{
	showmsg(msg,2);
	exit(1);
}

void signal_handler(int s)
{
	QFile::remove(tempFile);
	exit(s);
}

//******************************************************************************************************

PrintWrapper::PrintWrapper()
: QWidget(), stdin_set(false)
{
}

void PrintWrapper::slotPrint()
{
	KCmdLineArgs	*args = KCmdLineArgs::parsedArgs();

#if defined(HAVE_SIGACTION) && !defined(HAVE_SIGSET)
	struct sigaction action;
#endif /* HAVE_SIGACTION && !HAVE_SIGSET*/

	// read variables from command line
	QString	printer = args->getOption("d");
	QString	title = args->getOption("t");
	int	ncopies = QString(args->getOption("n")).toInt();
	QString	job_mode = args->getOption("j");
	QString	system = args->getOption("system");
	QCStringList	optlist = args->getOptionList("o");
	QMap<QString,QString>	opts;
	KURL::List	files;
	QStringList	filestoprint;
	stdin_set = args->isSet("stdin");
	bool	nodialog = !(args->isSet("dialog"));

	// parse options
	for (QCStringList::ConstIterator it=optlist.begin(); it!=optlist.end(); ++it)
	{
		QStringList	l = QStringList::split('=',QString(*it),false);
		if (l.count() >= 1) opts[l[0]] = (l.count() == 2 ? l[1] : QString::null);
	}

	// read file list
	for (int i=0; i<args->count(); i++)
		files.append(args->url(i));

	// some clean-up
	args->clear();

	// set default values if necessary
	if (job_mode == "console") job_output = 1;
	else if (job_mode == "none") job_output = 2;
	else job_output = 0;

	// some checking
	if (stdin_set && files.count() > 0)
	{
		showmsg(i18n("A file has been specified on the command line. Printing from STDIN will be disabled."), 1);
		stdin_set = false;
	}
	if (nodialog && !stdin_set && files.count() == 0)
	{
		errormsg(i18n("When using '--nodialog', you must at least specify one file to print or use the '--stdin' flag."));
	}

	KPrinter::ApplicationType	dialog_mode = (stdin_set || nodialog ? KPrinter::StandAlone : KPrinter::StandAlonePersistent);
	KPrinter::setApplicationType(dialog_mode);
	if (!stdin_set)
		KPrinter::addStandardPage(KPrinter::FilesPage);

	KPrinter	kprinter;
	if (nodialog)
	{
		KMPrinter	*prt(0);
		KMManager	*mgr = KMManager::self();

		mgr->printerList(false);
		if (!printer.isEmpty())
			prt = mgr->findPrinter(printer);
		else
			prt = mgr->defaultPrinter();

		if (prt == 0)
			errormsg(i18n("The specified printer or the default printer could not be found."));
		else if (!prt->autoConfigure(&kprinter))
			errormsg(i18n("Operation aborted."));
	}
	else if (!printer.isEmpty())
		kprinter.setSearchName(printer);
	kprinter.setDocName(title);
	kprinter.initOptions(opts);
	kprinter.setOption("kde-filelist", files.toStringList().join(", "));
	if (ncopies > 0)
		kprinter.setNumCopies(ncopies);

	if (nodialog)
		slotPrintRequested(&kprinter);
	else
	{
		KPrintDialog	*dlg = KPrintDialog::printerDialog(&kprinter, 0);
		if (dlg)
		{
			connect(dlg, SIGNAL(printRequested(KPrinter*)), SLOT(slotPrintRequested(KPrinter*)));
			dlg->exec();
			delete dlg;
		}
		else
			errormsg(i18n("Unable to construct the print dialog."));
	}

	QTimer::singleShot(10,kapp,SLOT(quit()));
}

void PrintWrapper::slotPrintRequested(KPrinter *kprinter)
{
	// download files if needed
	QStringList	files = QStringList::split(QRegExp(",\\s*"), kprinter->option("kde-filelist"), false), filestoprint;
	for (QStringList::ConstIterator it=files.begin(); it!=files.end(); ++it)
	{
		QString	tmpFile;
		KURL	url(*it);
		if (KIO::NetAccess::download(url, tmpFile))
		{
			filestoprint << tmpFile;
			if (kprinter->docName().isEmpty())
				kprinter->setDocName(url.fileName());
		}
	}

	if (kprinter->docName().isEmpty())
		kprinter->setDocName(stdin_set ? "<STDIN>" : "KPrinter");
	if (filestoprint.count() == 0)
	{
		// print from stdin
		if (!stdin_set)
		{
			showmsg(i18n("Can't print from STDIN: use '--stdin' option, or specify a file to print in the file selection area."), 2);
			return;
		}

#  if defined(HAVE_SIGSET)
		sigset(SIGHUP, signal_handler);
		sigset(SIGINT, signal_handler);
		sigset(SIGTERM, signal_handler);
#  elif defined(HAVE_SIGACTION)
		memset(&action, 0, sizeof(action));
		action.sa_handler = signal_handler;

		sigaction(SIGHUP, &action, NULL);
		sigaction(SIGINT, &action, NULL);
		sigaction(SIGTERM, &action, NULL);
#  else
		signal(SIGHUP, signal_handler);
		signal(SIGINT, signal_handler);
		signal(SIGTERM, signal_handler);
#  endif

		tempFile = locateLocal("tmp","kprinter_")+QString::number(getpid());
		filestoprint.append(tempFile);
		fromStdin = true;
		FILE	*fout = fopen(QFile::encodeName(filestoprint[0]),"w");
		if (!fout) errormsg(i18n("Unable to open temporary file."));
		char	buffer[8192];
		int	s;

		// read stdin and write to temporary file
		while ((s=fread(buffer,1,sizeof(buffer),stdin)) > 0)
			fwrite(buffer,1,s,fout);

		s = ftell(fout);
		fclose(fout);
		if (s <= 0)
		{
			showmsg(i18n("Stdin is empty, no job sent."), 2);
			QFile::remove(filestoprint[0]);
			return;
		}
	}
	else
		fromStdin = false;

	// print all files. Remove it after if printing from
	// stdin. "kprinter" shouldn't remove temp file itself,
	// otherwise the temp file might get removed before the
	// print process finishes.
	bool ok = kprinter->printFiles(filestoprint, fromStdin);

	if (!ok)
		showmsg(i18n("Error while printing files"), 2);
	else
	{
		QString	msg = i18n("<nobr>File(s) sent to printer <b>%1</b>.</nobr>").arg(kprinter->printerName());
		showmsg(msg,0);
	}
}

#include "printwrapper.moc"
